/*!
 * @file network_monitor.h
 *  a class to monitor the network: throughput, network nodes
 */

#ifndef _NETWORK_MONITOR_H
#define _NETWORK_MONITOR_H

#include "base_object.h"
#include "synchronizeable.h"
#include "shared_network_data.h"
#include "ip.h"

#include "network_node.h"

#include <list>

class NetworkStream;
class PeerInfo;


namespace OrxGui { class GLGuiBox; };



//!< a class that monitors the whole network. it tries to gather all informations its able to from the network
class NetworkMonitor : public Synchronizeable
{

  public:
    NetworkMonitor(NetworkStream* networkStream);
    virtual ~NetworkMonitor();

    /* client/masterserver/proxyserver addition/removal interface */
    void addNetworkNode(NetworkNode* node);
    void removeNetworkNode(NetworkNode* node);

    void addNode(PeerInfo* pInfo);
    void addNode(const IP& ip, int nodeType);
    void addNode(NetworkNode* node) { this->nodeList.push_back(node); }
    void addNode(NetworkNode* node, PeerInfo* pInfo);

    /** adds to @param node a network node @param pInfo a new client */
    inline void addClient(NetworkNode* node, PeerInfo* pInfo) { node->addClient(new NetworkNode(pInfo)); }
    /** adds to @param node a network node @param pInfo a new proxy server */
    inline void addActiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->addActiveProxyServer(new NetworkNode(pInfo)); }
    /** adds to @param node a network node @param pInfo a new proxy server */
    inline void addPassiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->addPassiveProxyServer(new NetworkNode(pInfo)); }
    /** adds to @param node a network node @param pInfo a new master server*/
    inline void addMasterServer(NetworkNode* node, PeerInfo* pInfo) { node->addMasterServer(new NetworkNode(pInfo)); }

    void removeNode(PeerInfo* pInfo);
    void removeNode(NetworkNode* node, PeerInfo* pInfo);

    inline void removeClient(NetworkNode* node, int userId) { node->removeClient(userId); }
    inline void removeActiveProxyServer(NetworkNode* node, int userId) { node->removeActiveProxyServer(userId); }
    inline void removePassiveProxyServer(NetworkNode* node, int userId) { node->removePassiveProxyServer(userId); }
    inline void removeMasterServer(NetworkNode* node, int userId) { node->removeMasterServer(userId); }

    PeerInfo* getFirstChoiceProxy() const;
    PeerInfo* getSecondChoiceProxy() const;
    /** @returns the local node */
    inline NetworkNode* getLocalNode() const { return this->localNode; };

    /** @returns the active proxy server list of the localnode */
    inline std::list<NetworkNode*> getActiveProxyServer() const { return this->localNode->getActiveProxyServer(); }

    /* slots admin and info interface */
    /** @returns the total number of players in this game (including all proxy servers etc)*/
    inline int getPlayerNumber() const { return this->playerNumber; }

    /** @returns true if there are still free network slots available at the local node*/
    inline bool gotFreeSlots() const { return (this->localNode->getPlayerNumber() < SharedNetworkData::getInstance()->getMaxPlayer()); }
    /** @param node node to be checked for slots @returns true if there are still free network slots available at this node */
    inline bool gotFreeSlots(NetworkNode* node) const { return (node->getPlayerNumber() < SharedNetworkData::getInstance()->getMaxPlayer()); }

    /** @returns true if the next client should be reconnected to some other proxy server with more connections */
    inline bool isReconnectNextClient() const { return (this->localNode->getPlayerNumber() >= SharedNetworkData::getInstance()->getMaxPlayer()); }

    inline const std::list<NetworkNode*>& getNodeList() const { return this->nodeList; };
    PeerInfo* getPeerByUserId( int userId);
    NetworkNode* getNodeByUserId( int userId);

    /* forced reconnection interface */
    inline void setForcedReconnection(IP address) { this->bForcedRecon = true; this->forcedReconnection = address;}
    inline bool isForcedReconnection() const { return this->bForcedRecon; }
    inline IP getForcedReconnectionIP() { this->bForcedRecon = false; return this->forcedReconnection; }


    void toggleGUI();

    void process();
    void debug() const;


  private:
    NetworkStream*               networkStream;              //!< reference to the one networkstream of the network
    NetworkNode*                 localNode;                  //!< reference to the local node
    std::list<NetworkNode*>      nodeList;                   //!< list of all network nodes

    OrxGui::GLGuiBox*            box;

    int                          playerNumber;                 //!< total number of players in the game
    int                          connectionNumber;             //!< total number of connections at this localhost

    IP                           forcedReconnection;           //!< ip of a forced reconnection
    bool                         bForcedRecon;                 //!< true if there is a forced recon
};


#endif
