/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli (patrick@orxonox.ethz.ch)
   co-programmer: Christoph Renner (rennerc@ee.ethz.ch)
*/


/* this is for debug output. It just says, that all calls to PRINT() belong to the DEBUG_MODULE_NETWORK module
   For more information refere to https://www.orxonox.net/cgi-bin/trac.cgi/wiki/DebugOutput
*/
#define DEBUG_MODULE_NETWORK

#include "class_list.h"
#include "debug.h"
#include "shell_command.h"

/* include your own header */
#include "network_manager.h"
#include "shared_network_data.h"
#include "network_stream.h"
#include "preferences.h"
#include "network_log.h"
#include "network_game_manager.h"
#include "proxy/proxy_control.h"


/* using namespace std is default, this needs to be here */


SHELL_COMMAND(debug, NetworkManager, debug);


NetworkManager* NetworkManager::singletonRef = NULL;

/**
 *  standard constructor
 */
NetworkManager::NetworkManager()
{
  /* set the class id for the base object */
  this->setClassID(CL_NETWORK_MANAGER, "NetworkManager");
  PRINTF(0)("START\n");

  /* initialize the references */
  this->networkStream = NULL;
  this->elapsedTime = 0.0f;


  int port = Preferences::getInstance()->getInt( "network", "telnetport", 0 );

  if ( port > 0 )
    NetworkLog::getInstance()->listen( port );

  PRINTF(0)("NetworkManager created\n");
}


/**
 *  standard deconstructor
 */
NetworkManager::~NetworkManager()
{
  PRINTF(0)("NetworkManager destructor\n");
  if ( this->networkStream )
  {
    delete this->networkStream;
    this->networkStream = NULL;
  }

  NetworkManager::singletonRef = NULL;
}


/**
 *  initializes the network manager
 */
void NetworkManager::initialize()
{
  PRINTF(0)("NetworkManager initzalized\n");
}


/**
 *  shutsdown the network manager
 */
void NetworkManager::shutdown()
{

}



/**
 *  creates a new NetworkStream of server type
 * @param clientPort: number of the TCP/UDP port for client connections
 * @param proxyPort: number of the TCP/UDP port for proxy connections
 */
int NetworkManager::createMasterServer(unsigned int port)
{
  // load the network settings
  NetworkSettings::getInstance()->loadData();

  // create the network stream
  this->networkStream = new NetworkStream(NET_MASTER_SERVER);
  this->networkStream->createServer( port, port + 1, port + 2);

  // start the network game manager
  this->networkStream->createNetworkGameManager();

  // init the proxy control center
  ProxyControl::getInstance();

  PRINTF(0)("Created Network Master Server\n");
  SDL_Delay(20);
  return 1;
}

/**
 *  creates a new network stream of proxy server type
 * @param port: number of the TCP port
 */
int NetworkManager::createProxyServer(unsigned int port)
{
  // load the network settings
  NetworkSettings::getInstance()->loadData();

  // create the network stream af
  this->networkStream = new NetworkStream(NET_PROXY_SERVER_ACTIVE );
  // first connect to the master server for synchronization
  this->networkStream->connectToMasterServer(NetworkSettings::getInstance()->getMasterAddr().ipString(), 10000);
  // start the handshake with the master server
  this->networkStream->startHandshake(NET_ID_MASTER_SERVER);

  // then start the server
  this->networkStream->createServer( port, port + 1, port + 2);

  // and to the other proxy servers also, this would be very nice if its works
  /* put it here....*/

  // init the proxy control center
  ProxyControl::getInstance();

  PRINTF(0)("Created Network Proxy Server\n");
  SDL_Delay(20);
  return 1;
}


/**
 *  creates a connection from one object to a host
 * @param hostName: the name of the destination host
 */
int NetworkManager::createClient(const std::string & name, unsigned int port)
{
  // load the network settings
  NetworkSettings::getInstance()->loadData();

  // create the network stream
  this->networkStream = new NetworkStream(NET_CLIENT);
  // connect to the master server, if a redirection should occure, this is handled in the NetworkStream itself
  this->networkStream->connectToMasterServer( name, port);


  // and start the handshake
  this->networkStream->startHandshake();
  // create the proxy control
  ProxyControl::getInstance();

  PRINTF(0)("Created Network Client\n");
  return 1;
}


/**
 * reconnects this client to another server
 * @param address new server address
 */
void NetworkManager::reconnectToServer(IP address)
{
  PRINTF(0)("Rec. reconnection command\n");
  this->networkStream->reconnectToServer(address);
}


/**
 * connects a synchronizeable to the network stream
 * @param sync: synchronizeable to connect
 */
void NetworkManager::connectSynchronizeable(Synchronizeable& sync)
{
  if( this->networkStream)
    this->networkStream->connectSynchronizeable(sync);
}


/**
 *  sync the network
 *  @param dtS: the seceonds elapsed since the last synchronize call
 */
void NetworkManager::synchronize( float dtS)
{
  this->elapsedTime += dtS;
  if( likely(this->elapsedTime < 1.0f / NETWORK_FREQUENCY))
    return;

  this->elapsedTime = 0.0f;

  if ( this->networkStream )
    networkStream->processData();

  NetworkGameManager::getInstance()->tick( this->elapsedTime );
}



/**
 * debug output
 */
 void NetworkManager::debug()
{
  PRINT(0)("=================Network::debug()=========\n");
  this->networkStream->debug();
  PRINT(0)("===========================================\n");
}
