/*!
 * @file network_node.h
 *  a class representing a node in the network (this can be a MASTER_SERVER, PROXY_SERVER or a CLIENT
 */

#ifndef _NETWORK_NODE_H
#define _NETWORK_NODE_H

#include "base_object.h"
#include "synchronizeable.h"
#include "peer_info.h"

#include <list>


//!< a class representing a node in the network (this can be a MASTER_SERVER, PROXY_SERVER or a CLIENT
class NetworkNode
{
  public:
    NetworkNode(PeerInfo* pInfo);
    ~NetworkNode();


    void addClient(NetworkNode* node);
    void addActiveProxyServer(NetworkNode* node);
    void addPassiveProxyServer(NetworkNode* node);
    void addMasterServer(NetworkNode* node);

    void removeClient(NetworkNode* node);
    void removeActiveProxyServer(NetworkNode* node);
    void removePassiveProxyServer(NetworkNode* node);
    void removeMasterServer(NetworkNode* node);

    void removeClient(int userId);
    void removeActiveProxyServer(int userId);
    void removePassiveProxyServer(int userId);
    void removeMasterServer(int userId);


    PeerInfo* getClient(int index) const;
    PeerInfo* getActiveProxyServer(int index) const;
    PeerInfo* getPassiveProxyServer(int index) const;
    PeerInfo* getMasterServer(int index) const;

    /** @returns the master server list */
    inline std::list<NetworkNode*> getMasterServer() const { return this->masterServerList; }
    /** @returns the active proxy server list */
    inline std::list<NetworkNode*> getActiveProxyServer() const { return this->activeProxyServerList; }
    /** @returns the passive proxy server list */
    inline std::list<NetworkNode*> getPassiveProxyServer() const { return this->passiveProxyServerList; }
    /** @returns the client list */
    inline std::list<NetworkNode*> getClient() const { return this->clientList; }

    PeerInfo* getPeerByUserId( int userId);

    /** @returns the number of players */
    inline int getPlayerNumber() const { return this->playerNumber; }
    /** @returns the node type of this node */
    inline int getNodeType() const { return this->peerInfo->nodeType; }
    /** @returns the peer info of this node */
    inline PeerInfo* getPeerInfo() const { return this->peerInfo; }

    NetworkNode* getNodeByUserId( int userId);

    void debug(int depth) const;


  private:
    int                          playerNumber;                 //!< localy direct connected player number
    int                          connectionNumber;             //!< number of connections ( can but musn't be equal players)
    PeerInfo*                    peerInfo;                     //!< the peer information about this node

    /* network nodes directly connected to this node */
    std::list<NetworkNode*>         clientList;                   //!< list of all clients in the network
    std::list<NetworkNode*>         activeProxyServerList;        //!< list of all proxy servers in the network
    std::list<NetworkNode*>         passiveProxyServerList;       //!< list of all proxy servers in the network
    std::list<NetworkNode*>         masterServerList;             //!< list of all master servers in the network (should be 1!! :D)

};

#endif /* _NETWORK_NODE_H */
