/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "water.h"
#include "util/loading/factory.h"
#include "util/loading/load_param.h"

#include "grid.h"
#include "material.h"

#include "util/loading/resource_manager.h"
#include "shader.h"

#include "skybox.h"
#include "state.h"


#include "network_game_manager.h"



CREATE_FACTORY(Water, CL_WATER);


Water::Water(const TiXmlElement* root)
{
  this->setClassID(CL_WATER, "Water");
  this->toList(OM_ENVIRON);

  this->resX = this->resY = 10;
  this->sizeX = this->sizeY = 1.0f;
  this->height = 0.5f;
  this->grid = NULL;

  this->velocities = NULL;
  this->viscosity = 5;
  this->cohesion = .0000000001;

  if (root != NULL)
    this->loadParams(root);

  this->rebuildGrid();
  this->waterShader = (Shader*)ResourceManager::getInstance()->load("shaders/water.vert", SHADER, RP_GAME, "shaders/water.frag");

  // To test the Wave equation
  //this->wave(5.0,4.0, 1, 10);

  height_handle = registerVarId( new SynchronizeableFloat( &height, &height, "height", PERMISSION_MASTER_SERVER ) );
  resX_handle = registerVarId( new SynchronizeableUInt( &resX, &resX, "resX", PERMISSION_MASTER_SERVER ) );
  resY_handle = registerVarId( new SynchronizeableUInt( &resY, &resY, "resY", PERMISSION_MASTER_SERVER ) );
  sizeX_handle = registerVarId( new SynchronizeableFloat( &sizeX, &sizeX, "sizeX", PERMISSION_MASTER_SERVER ) );
  sizeY_handle = registerVarId( new SynchronizeableFloat( &sizeY, &sizeY, "sizeY", PERMISSION_MASTER_SERVER ) );
}

Water::~Water()
{
}

void Water::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "size", this, Water, setSize)
  .describe("the size of the WaterSurface")
  .defaultValues(1.0f, 1.0f);

  LoadParam(root, "resolution", this, Water, setResolution)
  .describe("sets the resolution of the water surface")
  .defaultValues(10, 10);

  LoadParam(root, "height", this, Water, setHeight)
  .describe("the height of the Waves")
  .defaultValues(0.5f);
}

/**
 * @brief rebuilds the Grid below the WaterSurface, killing all hight information
 *
 * This should be called on all subGrid changes except wave and tick.
 */
void Water::rebuildGrid()
{
  if (this->velocities != NULL)
  {
    assert (this->grid != NULL);
    for (unsigned int i = 0; i < this->grid->rows(); i++)
      delete[] this->velocities[i];
    delete[] this->velocities;
  }

  //   WE DO NOT NEED THIS AS IT IS DONE IN WORLDENTITY->setModel();
  //   if (this->grid != NULL)
  //     this->grid = NULL;

  this->grid = new Grid(this->sizeX, this->sizeY, this->resX, this->resY);
  this->velocities = new float*[this->resX];
  for (unsigned int i = 0; i < this->grid->rows(); i++)
  {
    this->velocities[i] = new float[this->resY];
    for (unsigned int j = 0; j < this->resY; j++)
      this->velocities[i][j] = 0.0;
  }
  this->setModel(this->grid, 0);
}

void Water::wave(float x, float y, float z, float force)
{
  unsigned int row = 0, column = 0;
  if (!this->posToGridPoint( x, z, row, column))
    return;
  this->grid->height(row, column) -= force / ((y - this->getAbsCoor().y) +.1);
}

/**
 * after this a rebuild() must be called
 */
void Water::setResolution(unsigned int resX, unsigned int resY)
{
  this->resX = resX;
  this->resY = resY;
}

/**
 * after this a rebuild() must be called
 */
void Water::setSize(float sizeX, float sizeY)
{
  this->sizeX = sizeX;
  this->sizeY = sizeY;
}

void Water::setHeight(float height)
{
  this->height = height;
}


/**
 * @brief calculated the Position in the Grid, this Point is nearest to
 * @param x, the x-position over the Grid
 * @param z: the z-Position(or y) over the Grid
 * @param row returns the row if not out of range
 * @param column returns the column if not out of range
 * @returns true if a valid point is found, false if any x or y are out of range
 */
bool Water::posToGridPoint(float x, float z, unsigned int& row, unsigned int& column)
{
  float lower = this->getAbsCoor().y - this->sizeY *.5;
  float left = this->getAbsCoor().x - this->sizeX *.5;
  if (x > left && x < left + this->sizeX)
    row = (unsigned int) ((x- left) / this->grid->gridSpacing());
  else return false;
  if (z > lower && z < lower + this->sizeY)
    column = (unsigned int)((z-lower) / this->grid->gridSpacing());
  else return false;
  return true;
}


void Water::draw() const
{
  assert (this->grid != NULL);
  {
    glPushAttrib(GL_ENABLE_BIT);
    glEnable(GL_TEXTURE_2D);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    /* translate */
    glTranslatef (this->getAbsCoor ().x,
                  this->getAbsCoor ().y,
                  this->getAbsCoor ().z);
    Vector tmpRot = this->getAbsDir().getSpacialAxis();
    glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );

     if (State::getSkyBox())
     {
       glBindTexture(GL_TEXTURE_2D, State::getSkyBox()->getTexture(SKY_RIGHT));
       glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

       glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
       glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
       glEnable(GL_TEXTURE_GEN_S);
       glEnable(GL_TEXTURE_GEN_T);
     }
     glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

//    SkyBox::enableCubeMap();
    this->grid->draw();
    // this->waterShader->activateShader();
    //  this->waterMaterial->select();
    //Shader::deactivateShader();

    SkyBox::disableCubeMap();
    glDisable(GL_TEXTURE_GEN_S);
    glDisable(GL_TEXTURE_GEN_T);

    glPopMatrix();
    glPopAttrib();
  }
}

void Water::tick(float dt)
{
  ObjectManager::EntityList entityList = State::getObjectManager()->getObjectList(OM_GROUP_01_PROJ);
  ObjectManager::EntityList::iterator entity = entityList.begin();
  while (entity != entityList.end())
  {
    this->wave((*entity)->getAbsCoor(), 10.0*dt);
    entity++;
  }


  if (unlikely(this->velocities == NULL))
    return;
  /*
      THE OLD USELESS ALGORITHM
    phase += dt *.1;
    for (unsigned int i = 0; i < this->grid->rows(); i++)
    {
      for (unsigned int j = 0; j < this->grid->columns(); j++)
      {
        this->grid->height(i,j) = this->height*sin(((float)i/(float)this->grid->rows() *phase)+
            this->height*cos((float)j/(float)this->grid->columns()) * phase * 2.0);
      }
    }
    this->grid->rebuildNormals(this->height);*/


  unsigned int i, j;
  float u;

  // wave/advection
  // calc movement
  for(j = 1; j < this->grid->rows() - 1; j++)
  {
    for(i = 1; i < this->grid->columns() - 1; i++)
    {
      u =  this->grid->height(i+1,j)+ this->grid->height(i-1, j) +
           this->grid->height(i, j+1) + this->grid->height(i, j-1) -
           4 * this->grid->height(i, j);
      this->velocities[i][j] += dt * this->viscosity * this->viscosity * u / this->height;
      this->grid->height(i, j) += dt * this->velocities[i][j]  /* + dt * this->cohesion * u / this->height;*/;
      this->grid->height(i, j) *= .99;
    }
  }
  // boundraries
  for (j = 0; j < this->grid->rows(); j++)
  {
    this->grid->height(0,j) = this->grid->height(1,j);
    this->grid->height(this->grid->rows()-1,j) = this->grid->height(this->grid->rows()-2, j);
  }
  for (i = 0; i < this->grid->rows(); i++)
  {
    this->grid->height(i,0) = this->grid->height(i,1);
    this->grid->height(i,this->grid->columns()-1) = this->grid->height(i, this->grid->columns()-2);
  }
  /*
  for(j = 1; j < this->grid->rows() - 1; j++) {
      for(i = 1; i < this->grid->columns() - 1; i++) {
        this->grid->height(i, j) += dt * this->velocities[i][j];
      }
    }*/

  // calc normals
  //   float l[3];
  //   float m[3];
  //   for(j = 1; j < this->grid->rows() -1; j++) {
  //     for(i = 1; i < this->grid->columns() - 1; i++) {
  //       l[0] = this->grid->vertexG(i, j-1).x - this->grid->vertexG(i, j+1).x;
  //       l[1] = this->grid->vertexG(i, j-1).y - this->grid->vertexG(i, j+1).y;
  //       l[2] = this->grid->vertexG(i, j-1).z - this->grid->vertexG(i, j+1).z;
  //       m[0] = this->grid->vertexG(i-1,j).x - this->grid->vertexG(i+1, j).x;
  //       m[1] = this->grid->vertexG(i-1,j).y - this->grid->vertexG(i+1, j).y;
  //       m[2] = this->grid->vertexG(i-1,j).z - this->grid->vertexG(i+1, j).z;
  //       this->grid->normalG(i, j).x = l[1] * m[2] - l[2] * m[1];
  //       this->grid->normalG(i, j).y = l[2] * m[0] - l[0] * m[2];
  //       this->grid->normalG(i, j).z = l[0] * m[1] - l[1] * m[0];
  //     }
  //   }
  this->grid->rebuildNormals(this->height);
}



/**
 * function to handle changes in synced vars
 * @param id ids which have changed
 */
void Water::varChangeHandler( std::list< int > & id )
{
  if ( std::find( id.begin(), id.end(), height_handle ) != id.end() ||
       std::find( id.begin(), id.end(), resX_handle ) != id.end() ||
       std::find( id.begin(), id.end(), resY_handle ) != id.end() ||
       std::find( id.begin(), id.end(), sizeX_handle ) != id.end() ||
       std::find( id.begin(), id.end(), sizeY_handle ) != id.end()
     )
  {
    this->rebuildGrid();
  }

  WorldEntity::varChangeHandler( id );
}
