/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/


#include "network_node.h"

#include "debug.h"


/**
 * constructor
 */
NetworkNode::NetworkNode(PeerInfo* pInfo)
{
  this->playerNumber = 0;
  this->peerInfo = pInfo;
}


/**
 * deconstructor
 */
NetworkNode::~NetworkNode()
{}


/**
 * adds a client
 */
void NetworkNode::addClient(NetworkNode* node)
{
  this->clientList.push_back(node);
  this->playerNumber++;
  this->connectionNumber++;
}

/**
 * adds a proxy server
 */
void NetworkNode::addActiveProxyServer(NetworkNode* node)
{
  this->activeProxyServerList.push_back(node);
  this->connectionNumber++;
}

/**
 * adds a proxy server
 */
void NetworkNode::addPassiveProxyServer(NetworkNode* node)
{
  this->passiveProxyServerList.push_back(node);
}

/**
 * adds a master server
 */
void NetworkNode::addMasterServer(NetworkNode* node)
{
  this->masterServerList.push_back(node);
  this->playerNumber++;
}

/**
 * removes a client
 */
void NetworkNode::removeClient(NetworkNode* node)
{
  std::list<NetworkNode*>::iterator it = this->clientList.begin();
  for(; it != this->clientList.end(); it++)
  {
    if( *it == node)
    {
      this->clientList.erase(it);
      this->playerNumber--;
      return;
    }
  }

  PRINTF(2)("Could not remove client from the list, very strange...");
}

/**
 * removes a proxy server
 */
void NetworkNode::removeActiveProxyServer(NetworkNode* node)
{
  std::list<NetworkNode*>::iterator it = this->activeProxyServerList.begin();
  for(; it != this->activeProxyServerList.end(); it++)
  {
    if( *it == node)
    {
      this->activeProxyServerList.erase(it);
      this->playerNumber--;
      return;
    }
  }

  PRINTF(2)("Could not remove proxy server from the list, very strange...");
}

/**
 * removes a proxy server
 */
void NetworkNode::removePassiveProxyServer(NetworkNode* node)
{
  std::list<NetworkNode*>::iterator it = this->passiveProxyServerList.begin();
  for(; it != this->passiveProxyServerList.end(); it++)
  {
    if( *it == node)
    {
      this->passiveProxyServerList.erase(it);
      return;
    }
  }

  PRINTF(2)("Could not remove proxy server from the list, very strange...");
}

/**
 * removes a master server
 */
void NetworkNode::removeMasterServer(NetworkNode* node)
{
  std::list<NetworkNode*>::iterator it = this->masterServerList.begin();
  for(; it != this->masterServerList.end(); it++)
  {
    if( *it == node)
    {
      this->masterServerList.erase(it);
      this->playerNumber--;
      return;
    }
  }

  PRINTF(2)("Could not remove client from the list, very strange...");
}




/**
 * removes a client
 */
void NetworkNode::removeClient(int userId)
{
  std::list<NetworkNode*>::iterator it = this->clientList.begin();
  for(; it != this->clientList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
    {
      this->clientList.erase(it);
      this->playerNumber--;
      return;
    }
  }

  PRINTF(2)("Could not remove client from the list, very strange...");
}

/**
 * removes a proxy server
 */
void NetworkNode::removeActiveProxyServer(int userId)
{
  std::list<NetworkNode*>::iterator it = this->activeProxyServerList.begin();
  for(; it != this->activeProxyServerList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
    {
      this->activeProxyServerList.erase(it);
      this->playerNumber--;
      return;
    }
  }

  PRINTF(2)("Could not remove proxy server from the list, very strange...");
}

/**
 * removes a proxy server
 */
void NetworkNode::removePassiveProxyServer(int userId)
{
  std::list<NetworkNode*>::iterator it = this->passiveProxyServerList.begin();
  for(; it != this->passiveProxyServerList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
    {
      this->passiveProxyServerList.erase(it);
      return;
    }
  }

  PRINTF(2)("Could not remove proxy server from the list, very strange...");
}

/**
 * removes a master server
 */
void NetworkNode::removeMasterServer(int userId)
{
  std::list<NetworkNode*>::iterator it = this->masterServerList.begin();
  for(; it != this->masterServerList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
    {
      this->masterServerList.erase(it);
      this->playerNumber--;
      return;
    }
  }

  PRINTF(2)("Could not remove client from the list, very strange...");
}





/**
 *  gets the peer info by user id
 * @param userId  the user id of the node to look up
 * @return the peer info of this node NULL if nothing found
 */
PeerInfo* NetworkNode::getPeerByUserId( int userId)
{
  // look through the master server lists
  std::list<NetworkNode*>::const_iterator it = this->masterServerList.begin();
  for( ;it != this->masterServerList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
      return (*it)->getPeerInfo();
  }

  // look through the active proxy server list
  it = this->activeProxyServerList.begin();
  for( ; it != this->activeProxyServerList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
      return (*it)->getPeerInfo();
  }

  // look through the passive server list
  it = this->passiveProxyServerList.begin();
  for( ; it != this->passiveProxyServerList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
      return (*it)->getPeerInfo();
  }

  // look through the client list
  it = this->clientList.begin();
  for( ; it != this->clientList.end(); it++)
  {
    if( (*it)->getPeerInfo()->userId == userId)
      return (*it)->getPeerInfo();
  }

  return NULL;
}


/**
 * @param index index to the client
 * @return the client in the list or NULL if none
 */
PeerInfo* NetworkNode::getClient(int index) const
{
  if( (int)this->clientList.size() < index)
    return NULL;

  std::list<NetworkNode*>::const_iterator it = this->clientList.begin();
  for(int i = 0; it != this->clientList.end(); it++, i++)
  {
  if( i == index)
    return (*it)->getPeerInfo();
  }

  return NULL;
}


/**
 * @param index index to the client
 * @return the active proxy server in the list or NULL if none
 */
PeerInfo* NetworkNode::getActiveProxyServer(int index) const
{
  if( (int)this->activeProxyServerList.size() < index)
    return NULL;

  std::list<NetworkNode*>::const_iterator it = this->activeProxyServerList.begin();
  for(int i = 0; it != this->activeProxyServerList.end(); it++, i++)
  {
    if( i == index)
      return (*it)->getPeerInfo();
  }

  return NULL;
}


/**
 * @param index index to the client
 * @return the passive proxy server in the list or NULL if none
 */
PeerInfo* NetworkNode::getPassiveProxyServer(int index) const
{
  if( (int)this->passiveProxyServerList.size() < index)
    return NULL;

  std::list<NetworkNode*>::const_iterator it = this->passiveProxyServerList.begin();
  for(int i = 0; it != this->passiveProxyServerList.end(); it++, i++)
  {
    if( i == index)
      return (*it)->getPeerInfo();
  }

  return NULL;
}


/**
 * @param index index to the client
 * @return the master server in the list or NULL if none
 */
PeerInfo* NetworkNode::getMasterServer(int index) const
{
  if( (int)this->masterServerList.size() < index)
    return NULL;

  std::list<NetworkNode*>::const_iterator it = this->masterServerList.begin();
  for(int i = 0; it != this->masterServerList.end(); it++, i++)
  {
    if( i == index)
      return (*it)->getPeerInfo();
  }

  return NULL;
}


/**
 * debug function
 * @param depth: depth in the tree
 */
void NetworkNode::debug(int depth) const
{
  char indent[depth +1];
  for( int i = 0; i < depth; i++) {     indent[i] = ' ';  }
  indent[depth] = '\0';

  PRINT(0)("%s + %s, with ip: %s\n", indent,  this->peerInfo->getNodeTypeString().c_str(), this->peerInfo->ip.ipString().c_str());
  std::list<NetworkNode*>::const_iterator it;
  if( !this->masterServerList.empty())
  {
//     PRINT(0)("%s + master servers: %i\n", indent, this->masterServerList.size());
    it = this->masterServerList.begin();

    for(; it != this->masterServerList.end(); it++)
    {
//       IP* ip = &(*it)->getPeerInfo()->ip;
//       PRINT(0)("%s     - ms, id: %i (%s)\n", indent, (*it)->getPeerInfo()->userId, ip->ipString().c_str());
      (*it)->debug(depth+1);
    }
  }

  if( !this->activeProxyServerList.empty())
  {
//     PRINT(0)("%s + proxy servers active: %i\n", indent, this->activeProxyServerList.size());
    it = this->activeProxyServerList.begin();

    for(; it != this->activeProxyServerList.end(); it++)
    {
//       IP* ip = &(*it)->getPeerInfo()->ip;
//       PRINT(0)("%s     - ps-a, id: %i (%s)\n", indent, (*it)->getPeerInfo()->userId, ip->ipString().c_str());
      (*it)->debug(depth+1);
    }
  }


  if( !this->passiveProxyServerList.empty())
  {
//     PRINT(0)("%s proxy servers passive: %i\n", indent, this->passiveProxyServerList.size());
    it = this->passiveProxyServerList.begin();

    for(; it != this->passiveProxyServerList.end(); it++)
    {
//       IP* ip = &(*it)->getPeerInfo()->ip;
//       PRINT(0)("%s     - ps-p, id: %i (%s)\n", indent, (*it)->getPeerInfo()->userId, ip->ipString().c_str());
      (*it)->debug(depth+1);
    }
  }

  if( !this->clientList.empty())
  {
//     PRINT(0)("%s clients: %i\n", indent, this->clientList.size());
    it = this->clientList.begin();

    for(; it != this->clientList.end(); it++)
    {
//       IP* ip = &(*it)->getPeerInfo()->ip;
//       PRINT(0)("%s     - client, id: %i (%s)\n", indent, (*it)->getPeerInfo()->userId, ip->ipString().c_str());
      (*it)->debug(depth+1);
    }
  }
}

