/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GUI

#include "glgui_box.h"
#include <cassert>
#include "debug.h"

namespace OrxGui
{
  /**
   * standard constructor
  */
  GLGuiBox::GLGuiBox (OrxGui::Orientation orientation)
  {
    this->init();

    this->setOrientation(orientation);
  }


  /**
   * standard deconstructor
  */
  GLGuiBox::~GLGuiBox()
  {
    // unpack all the widgets.
    while(!this->_children.empty())
    {
      /// not deleting children here.
      this->_children.front()->setParentWidget(NULL);
      this->_children.pop_front();
    }
  }

  /**
   * initializes the GUI-element
   */
  void GLGuiBox::init()
  {
    this->setClassID(CL_GLGUI_BOX, "GLGuiBox");
  }

  void GLGuiBox::pack(GLGuiWidget* widget)
  {
    assert (widget != NULL);
    this->_children.push_back(widget);

    this->packing(widget);
  }


  void GLGuiBox::pack(GLGuiWidget* widget, std::list<GLGuiWidget*>::iterator pos)
  {
    this->_children.insert(pos, widget);
    this->packing(widget);
  }

  void GLGuiBox::pack(GLGuiWidget* widget, unsigned int position)
  {
    if (this->_children.empty())
      this->pack(widget);

    unsigned int pos = 0;
    std::list<GLGuiWidget*>::iterator it = this->_children.begin();

    for (; pos < position; ++pos)
    {
      if (this->_children.end() == ++it)
      {
        PRINTF(2)("Reached end of packing list, without getting to the designated position %d (i am at %d)\n", position, pos);
        this->pack(widget);
      }
    }
    this->_children.insert(it, widget);
    this->packing(widget);
  }

  void GLGuiBox::pack(GLGuiWidget* widget, GLGuiWidget* widgetPointer)
  {
    assert (widget != NULL && widgetPointer != NULL);

    std::list<GLGuiWidget*>::iterator it = this->_children.begin();
    for (; it != this->_children.end(); ++it)
    {
      if (widgetPointer == *it)
      {
        this->_children.insert(it, widget);
        return;
      }
    }
    PRINTF(2)("WidgetPointer %p not found, inserting at the end\n", widgetPointer);
    this->pack(widget);
    this->packing(widget);
  }

  void GLGuiBox::packFront(GLGuiWidget* widget)
  {
    this->_children.push_front(widget);
    this->packing(widget);
  }

  void GLGuiBox::packBack(GLGuiWidget* widget)
  {
    this->pack(widget);
  }

  void GLGuiBox::packing(GLGuiWidget* widget)
  {
    widget->setParentWidget(this);
    this->resize();
  }

  void GLGuiBox::unpack(GLGuiWidget* widget)
  {
    assert(widget != NULL);

    std::list<GLGuiWidget*>::iterator delWidget = std::find(this->_children.begin(), this->_children.end(), widget);
    if (delWidget != this->_children.end())
    {
      (*delWidget)->setParentWidget(NULL);
      this->_children.erase(delWidget);
    }
    this->resize();
  }

  void GLGuiBox::clear()
  {
    this->_children.clear();
    this->resize();
  }

  void GLGuiBox::showAll()
  {
    std::list<GLGuiWidget*>::iterator itC = this->_children.begin();
    while (itC != this->_children.end())
    {
      if ((*itC)->isA(CL_GLGUI_CONTAINER))
        static_cast<GLGuiContainer*>(*itC)->showAll();
      else
        (*itC)->show();
      itC++;
    }

    this->show();
  }

  void GLGuiBox::hideAll()
  {
    std::list<GLGuiWidget*>::iterator itC = this->_children.begin();
    while (itC != this->_children.end())
    {
      if ((*itC)->isA(CL_GLGUI_CONTAINER))
        static_cast<GLGuiContainer*>(*itC)->hideAll();
      else
        (*itC)->hide();
      itC++;
    }

    this->hide();
  }

  void GLGuiBox::resize()
  {
    if (orientation() == OrxGui::Vertical)
    {
      float height = borderTop();
      float width = 0.0f;
      std::list<GLGuiWidget*>::iterator widget;

      // find out how big the Widgets are.
      for (widget = this->_children.begin(); widget != this->_children.end(); ++widget)
      {
        (*widget)->setRelCoor2D(borderLeft(), height);
        height += (*widget)->getSizeY2D();
        width = fmax(width, (*widget)->getSizeX2D());
      }

      width += borderLeft() + borderRight();
      height += borderBottom(); /* *2 done further up */

      this->setSize2D(width, height);
    }
    else
    {
      float height = borderTop();
      float width = borderLeft();
      std::list<GLGuiWidget*>::iterator widget;

      // find out how big the Widgets are.
      for (widget = this->_children.begin(); widget != this->_children.end(); ++widget)
      {
        (*widget)->setRelCoor2D(width, borderTop());
        height = fmax(height, (*widget)->getSizeY2D());
        width += (*widget)->getSizeX2D();
      }

      width += borderRight() ;
      height += borderBottom(); /* *2 done further up */

      this->setSize2D(width, height);
    }
    GLGuiWidget::resize();

    // resize everything.
    //for (widget = this->_children.begin(); widget != this->_children.end(); ++widget)
    //{}
  }

  /**
   * @brief draws the GLGuiBox
   */
  void GLGuiBox::draw() const
  {
    this->beginDraw();
    GLGuiWidget::draw();
    this->endDraw();
  }
}
