/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "hover.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "util/loading/factory.h"
//#include "util/loading/resource_manager.h"

#include "key_mapper.h"
#include "state.h"

#include "graphics_engine.h"

#include "debug.h"

CREATE_FACTORY(Hover, CL_HOVER);

#include "script_class.h"
CREATE_SCRIPTABLE_CLASS(Hover, CL_HOVER,
                        addMethod("hasPlayer", ExecutorLua0ret<Playable,bool>(&Playable::hasPlayer))
                        //Coordinates
                        ->addMethod("setAbsCoor", ExecutorLua3<PNode,float,float,float>(&PNode::setAbsCoor))
                        ->addMethod("getAbsCoorX", ExecutorLua0ret<PNode, float>(&PNode::getAbsCoorX))
                        ->addMethod("getAbsCoorY", ExecutorLua0ret<PNode, float>(&PNode::getAbsCoorY))
                        ->addMethod("getAbsCoorZ", ExecutorLua0ret<PNode, float>(&PNode::getAbsCoorZ))
                       );

/**
 *  destructs the hover, deletes alocated memory
 */
Hover::~Hover ()
{
  this->setPlayer(NULL);

  //if (this->hoverBuffer != NULL)
  //  ResourceManager::getInstance()->unload(this->hoverBuffer);
}

/**
 * loads a Hover information from a specified file.
 * @param fileName the name of the File to load the hover from (absolute path)
 */
Hover::Hover(const std::string& fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for Hover.\n", fileName.c_str());
    return;
  }

  this->loadParams(doc.RootElement());

  //load sound
  //if (this->hoverBuffer != NULL)
  //  ResourceManager::getInstance()->unload(this->hoverBuffer);
  //this->hoverBuffer = (OrxSound::SoundBuffer*)ResourceManager::getInstance()->load("sound/engine/hover.wav", WAV);

}

/**
 *  creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
Hover::Hover(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_HYPERBLASTER));

  cannon->setName("BFG");

  this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1 ,1);
  this->addWeapon(cannon, 0, 2);

  this->getWeaponManager().changeWeaponConfig(1);
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

  this->loadModel("models/ships/hoverglider_mainbody.obj");
}


/**
 * initializes a Hover
 */
void Hover::init()
{

  //this->hitEntity = NULL;
  //this->hoverBuffer = NULL;

  //  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_HOVER, "Hover");
  this->toReflectionList();

  this->loadModel("models/ships/hoverglider_wing.obj", 1.0f, 3);
  this->loadModel("models/ships/hoverglider_rotor.obj", 1.0f, 4);
  this->loadModel("models/ships/rotor.obj", .45f, 5);

  bForward = bBackward = bLeft = bRight = bAscend = bDescend = false;
  mouseSensitivity = 0.005;

  this->rotorSpeed = 1000.0f;
  this->rotorCycle = 0.0f;
  this->cameraLook = 0.0f;
  this->rotation = 0.0f;
  this->acceleration = 15.0f;
  this->airFriction = 3.0f;

  // camera - issue
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  //this->cameraNode.setParentMode(PNODE_ROTATE_MOVEMENT);
  this->cameraNode.setParent(this);

  // rotors
  this->wingNodeLeft.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT );
  this->wingNodeLeft.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  this->wingNodeLeft.setParent(this);
  this->wingNodeLeft.setRelCoor(-1.5, -.3, -1.0);
  this->rotorNodeLeft.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->rotorNodeLeft.setParent(&this->wingNodeLeft);
  this->rotorNodeLeft.setRelCoor(0, 1.0, -2.3);

  this->wingNodeRight.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->wingNodeRight.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  this->wingNodeRight.setParent(this);
  this->wingNodeRight.setRelCoor(-1.5, -0.3, 1.0);
  this->rotorNodeRight.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->rotorNodeRight.setParent(&this->wingNodeRight);
  this->rotorNodeRight.setRelCoor(0, 1.0, 2.3);

  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);


  //add events to the eventlist
  registerEvent(KeyMapper::PEV_FORWARD);
  registerEvent(KeyMapper::PEV_BACKWARD);
  registerEvent(KeyMapper::PEV_LEFT);
  registerEvent(KeyMapper::PEV_RIGHT);
  registerEvent(KeyMapper::PEV_UP);
  registerEvent(KeyMapper::PEV_DOWN);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  registerEvent(EV_MOUSE_MOTION);


  // WEAPON_MANAGER configuration
  this->getWeaponManager().setSlotCount(5);

  this->getWeaponManager().setSlotPosition(0, Vector(-0.28, 1.186, -2.750), &this->wingNodeLeft);
  this->getWeaponManager().setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager().setSlotPosition(1, Vector(-0.28, 1.186, 2.750), &this->wingNodeRight);
  this->getWeaponManager().setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager().setSlotPosition(2, Vector(-1.63, .809, -.003));
  this->getWeaponManager().setSlotCapability(2, WTYPE_HEAVY);

  /// TODO: THESE ARE TOO MUCH
  this->getWeaponManager().setSlotPosition(3, Vector(-1.63, .678, -.652));
  this->getWeaponManager().setSlotDirection(3, Quaternion(-24/180 * M_PI, Vector(1,0,0)));

  this->getWeaponManager().setSlotPosition(4, Vector(-1.63, .678, .652));
  this->getWeaponManager().setSlotDirection(4, Quaternion(24/180 * M_PI, Vector(1,0,0)));

  this->cameraNode.setRelCoor(1,5,0);
  this->getWeaponManager().getFixedTarget()->setParent(&this->cameraNode);
  this->getWeaponManager().getFixedTarget()->setRelCoor(1000,0,0);

  // NETWORK THINGS

  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bAscend, &bAscend, "bAscend", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bDescend, &bDescend, "bDescend", PERMISSION_OWNER ) );

  registerVar( new SynchronizeableFloat( &rotation, &rotation, "rotation", PERMISSION_OWNER ) );
}

/**
 * loads the Settings of a Hover from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void Hover::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);
}


void Hover::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( true);

  State::getCameraNode()->setParentSoft(&this->cameraNode);
  State::getCameraNode()->setRelCoorSoft(-10, 0,0);
  State::getCameraTargetNode()->setParentSoft(&this->cameraNode);

  //this->soundSource.play(this->hoverBuffer, 0.3f, true);

}

void Hover::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
  this->detachCamera();

}


/**
 *  effect that occurs after the Hover is spawned
*/
void Hover::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the hover left the game
*/
void Hover::leftWorld ()
{}

/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void Hover::collidesWith(WorldEntity* entity, const Vector& location)
{
  Playable::collidesWith(entity, location);
}



/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void Hover::tick (float dt)
{
  Playable::tick(dt);

  // spaceship controlled movement
  this->movement(dt);
  this->rotorCycle += this->rotorSpeed * dt;

  this->debugNode(1);
}

/**
 *  calculate the velocity
 * @param time the timeslice since the last frame
*/
void Hover::movement (float dt)
{
  Vector accel(0.0, 0.0, 0.0);

  if( this->bForward )
  {
    accel += Vector(this->acceleration, 0, 0);
  }

  if( this->bBackward )
  {
    accel -= Vector(this->acceleration, 0, 0);
  }
  if( this->bLeft)
  {
    accel -= Vector(0, 0, this->acceleration);
  }

  if( this->bRight)
  {
    accel += Vector(0, 0, this->acceleration);
  }

  if (this->bAscend )
  {
    accel += Vector(0, this->acceleration, 0);
  }
  if (this->bDescend )
  {
    accel -= Vector(0, this->acceleration, 0);
  }

  Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);

  // this is the air friction (necessary for a smooth control)
  Vector damping = (this->velocity * this->airFriction);


  this->velocity += (accelerationDir - damping)* dt;

  this->shiftCoor (this->velocity * dt);

    // limit the maximum rotation speed.
  if (this->rotation != 0.0f)
  {
    float maxRot = 10.0 * dt;
    if (unlikely(this->rotation > maxRot)) this->rotation = maxRot;
    if (unlikely(this->rotation < -maxRot)) this->rotation = -maxRot;
    this->direction *= Quaternion(-M_PI/4.0*this->rotation, Vector(0,1,0));
    this->rotation = 0.0f;
  }

  this->setRelDirSoft(this->direction * Quaternion(-cameraLook, Vector(0,0,1)), 5);

  this->wingNodeLeft.setRelDirSoft(Quaternion(accel.z * .03 +this->rotation, Vector(1,0,0)), 5);
  this->rotorNodeLeft.setRelDirSoft(Quaternion(-accel.x * .05+this->rotation + cameraLook, Vector(0,0,1)), 5);

  this->wingNodeRight.setRelDirSoft(Quaternion(accel.z * .03 +this->rotation, Vector(1,0,0)), 5);
  this->rotorNodeRight.setRelDirSoft(Quaternion(-accel.x*.05 -this->rotation + cameraLook, Vector(0,0,1)), 5);
}


void Hover::draw() const
{
  Vector tmpRot;
  WorldEntity::draw();

  glPushMatrix();
  /// LEFT SIDE
  glTranslatef (this->wingNodeLeft.getAbsCoor ().x,
                this->wingNodeLeft.getAbsCoor ().y,
                this->wingNodeLeft.getAbsCoor ().z);
  tmpRot = this->wingNodeLeft.getAbsDir().getSpacialAxis();
  glRotatef (this->wingNodeLeft.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->getModel(3)->draw();
  glPopMatrix ();

  glPushMatrix();
  glTranslatef (this->rotorNodeLeft.getAbsCoor ().x,
                this->rotorNodeLeft.getAbsCoor ().y,
                this->rotorNodeLeft.getAbsCoor ().z);
  tmpRot = this->rotorNodeLeft.getAbsDir().getSpacialAxis();
  glRotatef (this->rotorNodeLeft.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->getModel(4)->draw();
  glTranslatef(0,-1,0);
  glRotatef(this->rotorCycle, 0,1,0);
  this->getModel(5)->draw();
  glPopMatrix ();

  /// RIGHT SIDE
  glPushMatrix();
  glTranslatef (this->wingNodeRight.getAbsCoor ().x,
                this->wingNodeRight.getAbsCoor ().y,
                this->wingNodeRight.getAbsCoor ().z);
  tmpRot = this->wingNodeRight.getAbsDir().getSpacialAxis();
  glRotatef (this->wingNodeRight.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  glScalef(1,1,-1);
  this->getModel(3)->draw();
  glPopMatrix ();

  glPushMatrix();
  glTranslatef (this->rotorNodeRight.getAbsCoor ().x,
                this->rotorNodeRight.getAbsCoor ().y,
                this->rotorNodeRight.getAbsCoor ().z);
  tmpRot = this->rotorNodeRight.getAbsDir().getSpacialAxis();
  glRotatef (this->rotorNodeRight.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  glScalef(1,1,-1);
  this->getModel(4)->draw();
  glTranslatef(0,-1,0);
  glRotatef(this->rotorCycle, 0,1,0);
  this->getModel(5)->draw();
  glPopMatrix ();
}

/**
 * @todo switch statement ??
 */
void Hover::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
    this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_UP)
    this->bAscend = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_DOWN)
    this->bDescend = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == KeyMapper::PEV_FORWARD)
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_BACKWARD)
    this->bBackward = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    float xMouse, yMouse;
    xMouse = event.xRel*mouseSensitivity;
    yMouse = event.yRel*mouseSensitivity;

    // rotate the Player around the y-axis
    this->rotation += xMouse;

    this->cameraLook += yMouse;
    // rotate the Camera around the z-axis
    if (cameraLook > M_PI_4)
      cameraLook = M_PI_4;
    else if (cameraLook < -M_PI_4)
      cameraLook = -M_PI_4;
    //this->cameraNode.setRelDirSoft(this->direction,10);
  }
}
