/*!
 * @file message_manager.h
 * @brief Definition of MessageManager
*/

#ifndef _MESSAGE_MANAGER_H
#define _MESSAGE_MANAGER_H

#include "synchronizeable.h"

#include <map>
#include <list>

/*
  Protocol:
    int nacks
    int acks[1..nacks]
    int nmsg
    (
      int length
      int number
      int MessageType
      byte * data
    )[1..nmsg]
*/


//!< different message ids
enum MessageType
{
  TESTMESSAGEID                  = 1,         //!< for testing purposes
  MSGID_DELETESYNCHRONIZEABLE,                //!< message for sync deletion
  MSGID_PREFEREDTEAM,                         //!< change prefered team
  MSGID_CHANGENICKNAME,                       //!< change nicknames
  MSGID_CHATMESSAGE,                          //!< chat message
  MSGID_RESPAWN,                              //!< respawn message

  MSGID_FORWARDMESSAGE                        //!< forwarding a messag between proxy and master server
};


typedef bool (*MessageCallback)( MessageType messageType, byte * data, int dataLength, void * someData, int userId );

enum RecieverType
{
  RT_ALL_BUT_ME = 1,   //!< message is sent to all users but myself
  RT_ALL_ME,           //!< message is sent to all users
  RT_USER,             //!< message is only sent to reciever
  RT_NOT_USER,         //!< message is sent to all but reciever
  RT_SERVER            //!< message is sent to server only
};

//TODO implement priority handling
enum MessagePriority
{
  MP_HIGHBANDWIDTH = 1,  //!< fast and reliable but uses a lot of bandwidth
  MP_LOWBANDWIDTH,       //!< may take a long time to arrive. reliable
  MP_UNRELIABLE          //!< unreliable. low bandwidth
};

struct NetworkMessage
{
  MessageType        messageType;
  byte *           data;
  int              length;
  int              number;
  MessagePriority  priority;
};

struct MessageUserQueue
{
  std::list<NetworkMessage> messages;
  std::list<int>            toAck;
  std::list<int>            recievedMessages;
};

typedef std::map<int,MessageUserQueue> MessageQueue;



struct MessageHandler
{
  MessageCallback cb;
  MessageType       messageType;
  void *          someData;
};

typedef std::map<MessageType,MessageHandler> MessageHandlerMap;

//! A class for sending messages over network
class MessageManager : public Synchronizeable {
 protected:
   MessageManager();
 public:
   inline static MessageManager * getInstance(){ if (!singletonRef) singletonRef = new MessageManager();  return singletonRef; }

   virtual ~MessageManager();

   bool registerMessageHandler( MessageType messageType, MessageCallback cb, void * someData );

   void sendMessage( MessageType messageType, byte * data, int dataLength, RecieverType recieverType, int reciever, MessagePriority messagePriority );

   void initUser( int userId );


  private:
   virtual int getStateDiff( int userId, byte* data, int maxLength, int stateId, int fromStateId, int priorityTH );
   virtual int setStateDiff( int userId, byte* data, int length, int stateId, int fromStateId );
   virtual void cleanUpUser( int userId );
   virtual void handleSentState( int userId, int stateId, int fromStateId ){}
   virtual void handleRecvState( int userId, int stateId, int fromStateId ){}


 private:
   static MessageManager *   singletonRef;           //!< the singleton reference

   std::list<NetworkMessage> incomingMessageQueue;   //!< the incoming message buffer
   MessageQueue              outgoingMessageQueue;   //!< stores messages to send
   MessageHandlerMap         messageHandlerMap;      //!< contains handlers for messages

   int                       newNumber;              //!< used to create unique message numbers


};

#endif /* _PROTO_CLASS_H */
