/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christoph Renner
   co-programmer: ...
*/

#include "player_stats.h"

#include "class_list.h"
#include "src/lib/util/loading/factory.h"

#include "player.h"
#include "state.h"
#include "shared_network_data.h"

#include "converter.h"

#include "preferences.h"

#include "debug.h"
#include "shell_command.h"


CREATE_FACTORY(PlayerStats, CL_PLAYER_STATS);


/**
 * constructor
 */
PlayerStats::PlayerStats( int userId )
{
  init();

  this->userId = userId;
}

/**
 * constructor
 */
PlayerStats::PlayerStats( const TiXmlElement* root )
{
  init();
}

void PlayerStats::init( )
{
  this->setClassID( CL_PLAYER_STATS, "PlayerStats" );

  this->userId = 0;
  this->teamId = TEAM_NOTEAM;
  this->preferedTeamId = TEAM_NOTEAM;
  this->score = 0;
  this->playableClassId = 0;
  this->modelFileName = "";
  this->nickName = "Player";
  this->oldNickName = "Player";

  userId_handle = registerVarId( new SynchronizeableInt( &userId, &userId, "userId" ) );
  teamId_handle = registerVarId( new SynchronizeableInt( &teamId, &teamId, "teamId" ) );
  preferedTeamId_handle = registerVarId( new SynchronizeableInt( &preferedTeamId, &preferedTeamId, "preferedUserId" ) );
  score_handle = registerVarId( new SynchronizeableInt( &score, &score, "score" ) );
  playableClassId_handle = registerVarId( new SynchronizeableInt( &playableClassId, &playableClassId, "playableClassId") );
  playableUniqueId_handle = registerVarId( new SynchronizeableInt( &playableUniqueId, &playableUniqueId, "playableUniqueId" ) );
  modelFileName_handle = registerVarId( new SynchronizeableString( &modelFileName, &modelFileName, "modelFileName" ) );
  nickName_handler = registerVarId( new SynchronizeableString( &nickName, &nickName, "nickName" ) );

  MessageManager::getInstance()->registerMessageHandler( MSGID_CHANGENICKNAME, changeNickHandler, NULL );

  PRINTF(0)("PlayerStats created\n");
}


/**
 * standard deconstructor
 */
PlayerStats::~PlayerStats()
{
}


 /**
 * override this function to be notified on change
 * of your registred variables.
 * @param id id's which have changed
  */
void PlayerStats::varChangeHandler( std::list< int > & id )
{
  if ( std::find( id.begin(), id.end(), playableUniqueId_handle ) != id.end() )
  {
    this->setPlayableUniqueId( this->playableUniqueId );

    PRINTF(0)("uniqueID changed %d %d %d\n", userId, SharedNetworkData::getInstance()->getHostID(), getUniqueID());
  }

  if ( std::find( id.begin(), id.end(), nickName_handler ) != id.end() )
  {
    PRINTF(0)("user %s is now known as %s\n", oldNickName.c_str(), nickName.c_str());
    this->oldNickName = nickName;
  }

  if ( std::find( id.begin(), id.end(), teamId_handle ) != id.end() )
  {
    PRINTF(0)("user %s joins team %i\n", this->nickName.c_str(), this->teamId);
    this->setPreferedTeamIdHandler( this->teamId);
  }

}


/**
 * handler for setting the prefered team id
 * @param newTeamId: the new team id
 */
void PlayerStats::setPreferedTeamIdHandler( int newTeamId)
{

  if( this->playable == NULL)
  {
    PRINTF(0)("could not set prefered team, since the playable is not yet set\n");
    return;
  }

  this->playable->setTeam(newTeamId);
}



/**
 * get stats for user userId
 * @param userId user's id
 * @return stats assigned to user
 */
PlayerStats * PlayerStats::getStats( int userId )
{
  const std::list<BaseObject*> * list = ClassList::getList( CL_PLAYER_STATS );

  if ( !list )
  {
    return NULL;
  }

  for ( std::list<BaseObject*>::const_iterator it = list->begin(); it != list->end(); it++ )
  {
    if ( dynamic_cast<PlayerStats*>(*it)->getUserId() == userId )
    {
      return dynamic_cast<PlayerStats*>(*it);
    }
  }

  return NULL;
}

/**
 * set playable class id and set playable
 */
void PlayerStats::setPlayableUniqueId( int uniqueId )
{
  const std::list<BaseObject*> * list = ClassList::getList( CL_PLAYABLE );

  if ( !list )
  {
    this->playableUniqueId = uniqueId;
    return;
  }

  this->playable = NULL;
  for ( std::list<BaseObject*>::const_iterator it = list->begin(); it != list->end(); it++ )
  {
    if ( dynamic_cast<Playable*>(*it)->getUniqueID() == uniqueId )
    {
      this->playable = dynamic_cast<Playable*>(*it);
      //TODO when OM_PLAYERS is ticked add line:
      //this->playable->toList( OM_PLAYERS );
      break;
    }
  }

  if ( this->playable && userId == SharedNetworkData::getInstance()->getHostID() )
  {
    State::getPlayer()->setPlayable( this->playable );
  }

  this->playableUniqueId = uniqueId;
}

/**
 * get playable associated to player
 * @return playable associated to player
 */
Playable * PlayerStats::getPlayable()
{
  if ( playable )
    return playable;

  assert( playableUniqueId > 0 );

  setPlayableUniqueId( playableUniqueId );

  assert( playable );

  return playable;
}

/**
 * client sends server a message to change nick and server changes nick directly
 * @param nick new nickname
 */
void PlayerStats::setNickName( std::string nick )
{
  if ( SharedNetworkData::getInstance()->isMasterServer())
  {
    this->nickName = nick;
    PRINTF(0)("user %s is now known as %s\n", oldNickName.c_str(), nickName.c_str());
    oldNickName = nickName;
    return;
  }
  else
  {
    byte * data = new byte[nick.length()+INTSIZE];

    assert( Converter::stringToByteArray( nick, data, nick.length()+INTSIZE) == nick.length()+INTSIZE );

    MessageManager::getInstance()->sendMessage( MSGID_CHANGENICKNAME, data, nick.length()+INTSIZE, RT_SERVER, 0, MP_HIGHBANDWIDTH );
    return;
  }
}

bool PlayerStats::changeNickHandler( MessageId messageId, byte * data, int dataLength, void * someData, int userId )
{
  std::string newNick;
  int res = Converter::byteArrayToString( data, newNick, dataLength );

  if ( res != dataLength )
  {
    PRINTF(2)("invalid message size from user %d\n", userId);
    newNick = "invalid";
  }

  if ( PlayerStats::getStats( userId ) )
    PlayerStats::getStats( userId )->setNickName( newNick );

  return true;
}

void PlayerStats::shellNick( const std::string& newNick )
{
  if ( getStats( SharedNetworkData::getInstance()->getHostID() ) )
    getStats( SharedNetworkData::getInstance()->getHostID() )->setNickName( newNick );

  Preferences::getInstance()->setString( "multiplayer", "nickname", newNick );
}



void PlayerStats::deleteAllPlayerStats( )
{
  const std::list<BaseObject*> * list;

  while ( (list  = ClassList::getList( CL_PLAYER_STATS )) != NULL && list->begin() != list->end() )
    delete *list->begin();
}



ScoreList PlayerStats::getScoreList( )
{
  ScoreList result;

  const std::list<BaseObject*> * list = ClassList::getList( CL_PLAYER_STATS );

  if ( !list )
  {
    return result;
  }

  for ( std::list<BaseObject*>::const_iterator it = list->begin(); it != list->end(); it++ )
  {
    PlayerStats & stats = *dynamic_cast<PlayerStats*>(*it);

    TeamScoreList::iterator it = result[stats.getTeamId()].begin();

    while (  it != result[stats.getTeamId()].end() && stats.score > it->score )
    {
      it++;
    }

    PlayerScore score;
    score.name = stats.getNickName();
    score.score = stats.getScore();

    result[stats.getTeamId()].insert(it, score);
  }

  return result;
}
