/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/

#include "glgui.h"
#include "shell_command.h"

#include "network_stream.h"
#include "debug.h"

#include "proxy/network_settings.h"
#include "shared_network_data.h"

#include "network_monitor.h"
#include "network_node.h"
#include "peer_info.h"
#include "network_stream.h"
#include "netdefs.h"

#include <vector>


#include "network_stats_widget.h"

SHELL_COMMAND(gui, NetworkMonitor, toggleGUI)
 ->setAlias("ProxyGui");
SHELL_COMMAND(debug, NetworkMonitor, debug);



/**
 * starts a network monitor
 */
NetworkMonitor::NetworkMonitor(NetworkStream* networkStream)
  : Synchronizeable()
{
  this->setClassID(CL_NETWORK_MONITOR, "NetworkMonitor");

  this->networkStream = networkStream;
  this->playerNumber = 0;
  // create the localnode, init it and add it to the nodes list
  this->localNode = new NetworkNode( this->networkStream->getPeerInfo());
  this->addNode(this->localNode);

  this->setSynchronized(false);

  // read in the proxy server list, this is only the case for the master server
  if( SharedNetworkData::getInstance()->isMasterServer())
  {
    // assuming that the config files are already read we get the proxy servers
    std::vector<IP>* proxyList = NetworkSettings::getInstance()->getProxyList();
    std::vector<IP>::iterator it = proxyList->begin();
    // create a peer info class and a network node class for each new proxy and add them to the passive list
    for(; it < proxyList->end(); it++)
    {
      PeerInfo* peer = new PeerInfo();
      peer->ip = (*it);
      peer->nodeType = NET_PROXY_SERVER_ACTIVE;
      peer->userId = -1;

      NetworkNode* node = new NetworkNode(peer);
      this->addNode( node);
      this->addActiveProxyServer( this->localNode, peer);
    }
  }
  this->box = NULL;
}


/**
 * deconstructor
 */
NetworkMonitor::~NetworkMonitor()
{
  if( this->localNode)
    delete this->localNode;
}


/**
 * add the network node
 * @param node to add
 */
void NetworkMonitor::addNetworkNode(NetworkNode* node)
{
  this->nodeList.push_back(node);
}


/**
 * add the network node
 * @param node to add
 */
void NetworkMonitor::removeNetworkNode(NetworkNode* node)
{
  std::list<NetworkNode*>::iterator it = this->nodeList.begin();
  for(; it != this->nodeList.end(); it++)
  {
    if( *it == node)
    {
      if( node->getNodeType() == NET_CLIENT)
        this->playerNumber--;

      this->nodeList.erase(it);
      return;
    }
  }
}

/**
 * tihs adds the new network node
 * @param ip ip of the new node
 */
void NetworkMonitor::addNode(const IP& ip, int nodeType)
{
  PeerInfo* pInfo = new PeerInfo();
  pInfo->nodeType = nodeType;
  pInfo->ip = ip;

  this->addNode( pInfo);
}


/**
 * adds a network node to the local node
 *  @param pInfo node information
 */
void NetworkMonitor::addNode(PeerInfo* pInfo)
{
  if( this->localNode == NULL)
    return;

  if( pInfo->isClient())
    this->localNode->addClient(pInfo);
  else if( pInfo->isProxyServerActive())
  {
    this->localNode->addActiveProxyServer(pInfo);
    // create a new node, since a proxy can connect clients again
    NetworkNode* node = new NetworkNode(pInfo);
    this->nodeList.push_back(node);
  }
  else if( pInfo->isMasterServer())
  {
    this->localNode->addMasterServer(pInfo);
  }
}


/**
 * adds a network node to the local node
 *  @param node node to add to
 *  @param pInfo node information
 */
void NetworkMonitor::addNode(NetworkNode* node, PeerInfo* pInfo)
{
  if( node == NULL)
    return;

  if( pInfo->isClient())
    node->addClient(pInfo);
  else if( pInfo->isProxyServerActive())
    node->addActiveProxyServer(pInfo);
  else if( pInfo->isMasterServer())
    node->addMasterServer(pInfo);
}


/**
 * @returns the proxy server of the first choice
 */
PeerInfo* NetworkMonitor::getFirstChoiceProxy() const
{
  // return the fist proxy in the list
  return this->localNode->getActiveProxyServer(0);
}


/**
 * @returns the proxy server of second choice
 */
PeerInfo* NetworkMonitor::getSecondChoiceProxy() const
{
  // return the second server in the list
  return this->localNode->getActiveProxyServer(1);
}


/**
 * this displays the network monitor gui
 */
void NetworkMonitor::toggleGUI()
{
  if (this->box == NULL)
  {
    this->box = new OrxGui::GLGuiBox(OrxGui::Vertical);
    {
      NetworkStatsWidget* netStats = new NetworkStatsWidget(this);
      this->box->pack(netStats);

    }

    this->box->showAll();
    this->box->setAbsCoor2D(300, 40);
  }
  else
  {
    delete this->box;
    this->box = NULL;
  }
}

/**
 * processes the network monitor data
 */
void NetworkMonitor::process()
{
  this->playerNumber = 0;
  std::list<NetworkNode*>::iterator it = this->nodeList.begin();
  for(; it != this->nodeList.end(); it++)
  {
    this->playerNumber += (*it)->getPlayerNumber();
  }

  // check if a proxy server has to activated

//   this->debug();
}

/**
 * prints out the debug informations
 */
void NetworkMonitor::debug() const
{
  PRINT(0)("================================= Network Monitor::debug() =====\n");
  PRINT(0)(" I am: %s\n", this->localNode->getPeerInfo()->getNodeTypeString().c_str());
  PRINT(0)(" Total count of network connections: %i\n", this->connectionNumber);
  PRINT(0)(" Total count of players: %i\n", this->playerNumber);
  PRINT(0)(" Max players on this server: %i\n", SharedNetworkData::getInstance()->getMaxPlayer());

  std::list<NetworkNode*>::const_iterator it = this->nodeList.begin();
  for(; it != this->nodeList.end(); it++)
  {
    (*it)->debug(0);
  }

  PRINT(0)("================================================================\n");
}

