/*!
 * @file network_monitor.h
 *  a class to monitor the network: throughput, network nodes
 */

#ifndef _NETWORK_MONITOR_H
#define _NETWORK_MONITOR_H

#include "base_object.h"
#include "synchronizeable.h"
#include "shared_network_data.h"
#include "ip.h"

#include "network_node.h"

#include <list>

class NetworkStream;
class PeerInfo;


namespace OrxGui { class GLGuiBox; };



//!< a class that monitors the whole network. it tries to gather all informations its able to from the network
class NetworkMonitor : public Synchronizeable
{

  public:
    NetworkMonitor(NetworkStream* networkStream);
    virtual ~NetworkMonitor();

    /* client/masterserver/proxyserver addition/removal interface */
    void addNetworkNode(NetworkNode* node);
    void removeNetworkNode(NetworkNode* node);

    void addNode(PeerInfo* pInfo);
    void addNode(const IP& ip, int nodeType);
    void addNode(NetworkNode* node) { this->nodeList.push_back(node); }
    void addNode(NetworkNode* node, PeerInfo* pInfo);

    /** adds to @param node a network node @param pInfo a new client */
    inline void addClient(NetworkNode* node, PeerInfo* pInfo) { node->addClient(pInfo); }
    /** adds to @param node a network node @param pInfo a new proxy server */
    inline void addActiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->addActiveProxyServer(pInfo); }
    /** adds to @param node a network node @param pInfo a new proxy server */
    inline void addPassiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->addPassiveProxyServer(pInfo); }
    /** adds to @param node a network node @param pInfo a new master server*/
    inline void addMasterServer(NetworkNode* node, PeerInfo* pInfo) { node->addMasterServer(pInfo); }

    inline void removeClient(NetworkNode* node, PeerInfo* pInfo) { node->removeClient(pInfo); }
    inline void removeActiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->removeActiveProxyServer(pInfo); }
    inline void removePassiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->removePassiveProxyServer(pInfo); }
    inline void removeMasterServer(NetworkNode* node, PeerInfo* pInfo) { node->removeMasterServer(pInfo); }

    PeerInfo* getFirstChoiceProxy() const;
    PeerInfo* getSecondChoiceProxy() const;
    /** @returns the local node */
    inline NetworkNode* getLocalNode() const { return this->localNode; };

    /** @returns the active proxy server list of the localnode */
    inline std::list<PeerInfo*> getActiveProxyServer() const { return this->localNode->getActiveProxyServer(); }

    /* slots admin and info interface */
    /** @returns the total number of players in this game (including all proxy servers etc)*/
    inline int getPlayerNumber() const { return this->playerNumber; }

    /** @returns true if there are still free network slots available at the local node*/
    inline bool gotFreeSlots() const { return (this->localNode->getPlayerNumber() < SharedNetworkData::getInstance()->getMaxPlayer()); }
    /** @param node node to be checked for slots @returns true if there are still free network slots available at this node */
    inline bool gotFreeSlots(NetworkNode* node) const { return (node->getPlayerNumber() < SharedNetworkData::getInstance()->getMaxPlayer()); }

    /** @returns true if the next client should be reconnected to some other proxy server with more connections */
    inline bool isReconnectNextClient() const { return (this->localNode->getPlayerNumber() >= SharedNetworkData::getInstance()->getMaxPlayer()); }

    inline const std::list<NetworkNode*>& getNodeList() const { return this->nodeList; };

    void toggleGUI();

    void process();
    void debug();


  private:
    NetworkStream*               networkStream;              //!< reference to the one networkstream of the network
    NetworkNode*                 localNode;                  //!< reference to the local node
    std::list<NetworkNode*>      nodeList;                   //!< list of all network nodes

    OrxGui::GLGuiBox*            box;

    int                          playerNumber;                 //!< total number of players in the game
    int                          connectionNumber;             //!< total number of connections at this localhost
};


#endif
