/**
 * @file ip.cc
 * @brief A IP class, that handles IP names and resolution.
 *
 * code taken from audiere.
 *
 *
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
 * co-programmer: Patrick Boenzli
 */

#include "ip.h"

#include <iostream>

#include "multi_type.h"
#include "substring.h"

/**
 * @brief default constructor
 */
IP::IP()
{
  this->_host = 0;
  this->_port = IP::_defaultPort;
}


/**
 * @brief constructor from ip and port
 * @param ip the IP
 * @param port the Port
 * @return self
 */
IP::IP(int ip, int port)
{
  this->_host = ip;
  if (port != -1)
    this->_port = port;
  else
    this->_port = IP::_defaultPort;
}


/**
 * @brief constructor from a String
 * @param ip the IP as a String.
 * @param resolve if true, the IP is resolved via DNS,
 * @return self
 */
IP::IP(const std::string& ip, bool resolve)
{
  *this = IP::stringToIP(ip, IP::_defaultPort, resolve);
}

/**
 * @brief constructor from a String
 * @param ip the IP as a String.
 * @param port The port to be resolved.
 * @param resolve if true, the IP is resolved via DNS,
 * @return self
 */
IP::IP(const std::string& ip, int port, bool resolve)
{
  *this = IP::stringToIP(ip, port, resolve);
}


/**
 * @brief constructor from an SDL net structure IPaddress
 * @param ip the ip.
 * @return self
 */
IP::IP(const IPaddress& ip)
{
  this->_host = ip.host;
  this->_port = ip.port;
}


/**
 * @brief copy constructor.
 * @param ip the IP to copy.
 * @return self
 */
IP::IP(const IP& ip)
{
  *this = ip;
}


/**
 * @brief constructs an IP out of the four pairs and a Port.
 * @param first The first octal.
 * @param second The second octal.
 * @param third The third octal.
 * @param fourth The fourth octal.
 * @param port The Port.
 * @return self.
 */
IP::IP(unsigned int first, unsigned int second, unsigned int third, unsigned int fourth, int port)
{
  this->_host = (first << 24) +
                (second << 16) +
                (third <<  8) +
                fourth;
  this->_port = port;
}


/**
 * @brief copy operator
 * @param ip the IP to copy.
 * @return self.
 */
const IP& IP::operator=(const IP& ip)
{
  this->_host = ip.host();
  this->_port = ip.port();
  return *this;
}


/**
 * @brief copy operator
 * @param ip the IP to copy.
 * @return self.
 */
const IP& IP::operator=(const IPaddress& ip)
{
  this->_host = ip.host;
  this->_port = ip.port;
  return *this;
}


/**
 * @brief comparison operator
 * @param ip the IP to compare
 * @return true if ip _and_ port match.
 */
bool IP::operator==(const IP& ip) const
{
  return (this->_host == ip.host() &&
          this->_port == ip.port());
}


/**
 * @brief comparison operator
 * @param ip the IP to compare
 * @return true if ip _or_ port do not match.
 */
bool IP::operator!=(const IP& ip) const
{
  return (this->_host != ip.host() ||
          this->_port != ip.port());
}


/**
 * @brief converts a String into an IP Object.
 * @param ip The string holding an IP.
 * @param port The port to be resolved.
 * @param resolve if true, the IP is resolved via DNS,
 * otherwise (resolve == false) the IP is being transformed
 * from a String (xxx.xxx.xxx.xxx) to an integer.
 *
 * @return A newly constructed IP.
 */
IP IP::stringToIP(const std::string& ip, int port, bool resolve)
{
  if (resolve)
  {
    IPaddress ipaddr;

    SDLNet_ResolveHost(&ipaddr, ip.c_str(), port);

    return IP(ipaddr);
  }
  else
  {
    SubString ipaddr(ip, '.');
    if(ip.size() != 4 )
      return IP();

    MultiType part0(ipaddr[0]);
    MultiType part1(ipaddr[1]);
    MultiType part2(ipaddr[2]);
    MultiType part3(ipaddr[3]);

    int addr = (part0.getInt() << 24) +
               (part1.getInt() << 16) +
               (part2.getInt() <<  8) +
               part3.getInt();
    return IP(addr, port);
  }
}


/**
 * @brief if you want to have a specific part of an IP
 * @param part the n'th part of the IP addr (splitted by '.').
 * @return the amount held in the designated part.
 */
int IP::ipPart(unsigned int part) const
{
  switch (part)
  {
    case 0:
      return  (_host & 0xFF000000) >> 24;
    case 1:
      return  (_host & 0x00FF0000) >> 16;
    case 2:
      return  (_host & 0x0000FF00) >> 8;
    case 3:
      return  (_host & 0x000000FF);
    default:
      return -1;
  }
}


/**
 * @return the Ip as a string.
 */
std::string IP::ipString() const
{
  return IP::ipToString(this->_host, this->_port);
}


/**
 * @brief converts an IPaddress struct into a String.
 * @param ipaddr the IP address as a SDL_net struct.
 * @return the string retrieved from the IP.
 */
std::string IP::ipToString(const IPaddress& ipaddr)
{
  int ip = SDLNet_Read32 (ipaddr.host);
  return IP::ipToString(ip, ipaddr.port);
}


/**
 * @brief converts a IP into a String (without port).
 * @param host the IP to put into the string.
 * @param port -1 if not wanted
 * @return the string of the ip.
 */
std::string IP::ipToString(int host, int port)
{
  MultiType part0((int) ((host & 0xFF000000) >> 24));
  MultiType part1((int) ((host & 0x00FF0000) >> 16));
  MultiType part2((int) ((host & 0x0000FF00) >>  8));
  MultiType part3((int) ((host & 0x000000FF) ));

  std::string addr = part3.getString() + "." + part2.getString() + "." +
                     part1.getString() + "." + part0.getString();

  if (port != -1)
    addr += ":" + MultiType(port).getString();
  return addr;
}


/// default port definition.
int IP::_defaultPort = 9999;

/**
 * @brief if no IP is supplied this port is used, so that IP can be resolved usefully.
 * @param defaultPort The default port.
 */
void IP::setDefaultPort(int defaultPort)
{
  IP::_defaultPort = defaultPort;
}


/**
 * @brief print out the IP in a nice fashion
 */
void IP::debug() const
{
  std::cout << "IP is " << this->ipString() << std::endl;
}
