/*!
 * @file network_stream.h
 *  implementation of a network pipe
 */

#ifndef _NETWORK_STREAM
#define _NETWORK_STREAM

#include <vector>
#include <list>
#include <map>

#include "data_stream.h"
#include "server_socket.h"
#include "handshake.h"
#include "monitor/connection_monitor.h"
#include "udp_server_socket.h"
#include "peer_info.h"

#include "shared_network_data.h"

class Synchronizeable;
class NetworkSocket;
class ServerSocket;
class NetworkGameManager;
class NetworkMonitor;


typedef std::list<Synchronizeable*>  SynchronizeableList;
typedef std::map<int,PeerInfo>       PeerList;


class NetworkStream : public DataStream
{

  public:
    NetworkStream();
    NetworkStream(int nodeType);
    virtual ~NetworkStream();

    void init();

    void connectToMasterServer(std::string host, int port);
    void connectToProxyServer(std::string host, int port);
    void createServer(int port);

    void createNetworkGameManager();
    void startHandshake();

    /* synchronizeable interface */
    void connectSynchronizeable(Synchronizeable& sync);
    void disconnectSynchronizeable(Synchronizeable& sync);

    inline int getMaxConnections(){ return SharedNetworkData::getInstance()->getMaxPlayer(); }

    /* functions for the peerInfo information retreival */
    inline bool isUserIdActive( int userID ) { return (peers.find(userID) != peers.end()); }
    inline bool isUserMasterServer( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isMasterServer(); }
    inline bool isUserProxyServer( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isProxyServer(); }
    inline bool isUserClient( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isClient(); }

    /* peering interface */
    inline PeerInfo & getPeerInfo( int userId ) { return peers[userId]; }
    inline PeerInfo* getPeerInfo() { return this->pInfo; }
    inline PeerList getPeerInfoList() { return this->peers; }

    /* data processing*/
    virtual void processData();

    /* debugging */
    void debug();


  private:

    inline SynchronizeableList::const_iterator getSyncBegin(){ return synchronizeables.begin(); }
    inline SynchronizeableList::const_iterator getSyncEnd(){ return synchronizeables.end(); }
    void cleanUpOldSyncList();
    int getSyncCount();

    void updateConnectionList();

    /* handle processes */
    void handleHandshakes();
    void handleUpstream( int tick );
    void handleDownstream(int tick );

    /* handle events*/
    void handleNewClient( int userId );
    void handleReconnect( int userId );
    void handleDisconnect( int userId );

    void writeToNewDict( byte * data, int length, bool upstream );


  private:
    PeerList                   peers;                       //!< list of the network node informations

    PeerInfo*                  pInfo;                       //!< the info about the local peer node (not in the PeerList)

    std::list<int>             freeSocketSlots;             //!< list of free sockets (to ensure not to resycle sockets)
    int                        currentState;                //!< current state id

    NetworkMonitor*            networkMonitor;              //!< the network monitor
    NetworkGameManager*        networkGameManager;          //!< reference to the network game manager
    ServerSocket*              serverSocket;                //!< the listening socket of the server

    std::map<int,int>          oldSynchronizeables;         //!< used to save recently deleted sync ids to not recreate them
    SynchronizeableList        synchronizeables;            //!< list of the synchronizeables

    byte                       buf[UDP_PACKET_SIZE];        //!< used by handleUp/Downstream
    byte                       compBuf[UDP_PACKET_SIZE];    //!< used by handleUp/Downstream
    int                        remainingBytesToWriteToDict; //!< if > 0 NetworkStream will write packets to DATA/dicts/newdict

    int                        dictServer;                  //!< the zip dict for the server
    int                        dictClient;                  //!< the zip dict for the client
};
#endif /* _NETWORK_STREAM */
