/*!
 * @file glgui_widget.h
 * The gl_widget of the openglGUI
 */

#ifndef _GLGUI_WIDGET_H
#define _GLGUI_WIDGET_H

#include "element_2d.h"

#include "glgui_defs.h"

#include "material.h"
#include "font.h"

#include "rect2D.h"

#include "event.h"
#include "sigslot/signal.h"

namespace OrxGui
{

  class GLGuiCursor;


  //! This is widget part of the openglGUI class
  /**
   * A widget is the main class of all the elements of th GUI.
   */
  class GLGuiWidget : public Element2D
  {
  public:
    GLGuiWidget(GLGuiWidget* parent = NULL);
    virtual ~GLGuiWidget();

    void show();
    void hide();

    void setParentWidget(GLGuiWidget* parent);
    GLGuiWidget* parent() const { return this->_parent; }

    /// FOCUS
    /** @brief gives mouse - focus to this widget */
    void giveMouseFocus();
    void breakMouseFocus();

    /** @returns true if the widget is focusable */
    bool focusable() const { return this->_focusable; };
    /** @returns true if the position is inside of the Widget. @param position the position to check */
    bool focusOverWidget(const Vector2D& position) const;
    /** @brief overloaded function, that returns true if the cursor is on top of the Widget */
    bool focusOverWidget(const OrxGui::GLGuiCursor* const cursor) const;

    /** @returns the currently mouse - focused Widget (NULL if none is focused). */
    static GLGuiWidget* mouseFocused() { return GLGuiWidget::_mouseFocused; };

    /// SELECT
    void select();
    void unselect();
    /** @returns true if the Widget is selectable */
    bool selectable() const { return this->_selectable; }

    /** @returns the currently Selected Widget (NULL if none is selected). */
    static GLGuiWidget* selected() { return GLGuiWidget::_selected; };

    /// CLICK
    bool pushed() { return _pushed; };
    void click(const Vector2D& pos);
    void release(const Vector2D& pos);
    bool clickable() const { return this->_clickable; };

    OrxGui::State state() const { return this->_state; };

    Rect2D& backRect() { return this->_backRect; };
    const Rect2D& backRect() const { return this->_backRect; };

    void setFrontColor(const Color& frontColor, bool instantaniously = false);

    void setWidgetSize(const Vector2D& size);
    void setWidgetSize(float x, float y);

    void animateBack();

    ///////////////////////////////////////////////////////////////////////////////////
    /// STYLE /////////////////////////////////////////////////////////////////////////
    ////////////////////////////////
    /// Retrieve Current Values. ///
    ////////////////////////////////
    /** @returns current left borderWidth */
    inline float borderLeft() const { return _currentStyle._borderLeft; }
    /** @returns current right borderWidth */
    inline float borderRight() const { return _currentStyle._borderRight; }
    /** @returns current top borderWidth */
    inline float borderTop() const { return _currentStyle._borderTop; }
    /** @returns burrent bottom borderWidth */
    inline float borderBottom() const { return _currentStyle._borderBottom; }


    /** @returns current textSize */
    inline float textSize() const { return _currentStyle._textSize; }
    /** @returns the Background Color */
    inline const Color& backgroundColor() const { return _currentStyle._background.diffuseColor(); }
    /** @returns the current Background Material. */
    inline const Material& background() const { return _currentStyle._background; }
    /** @returns the current background Texture. */
    inline const Texture& backgroundTexture() const { return _currentStyle._background.diffuseTexture(); }
    /** @returns the current foreground Color */
    inline const Color& foregroundColor() const { return _currentStyle._foreground.diffuseColor(); }
    /** @returns the current ForeGroung Material. */
    inline const Material& foreground() const { return _currentStyle._foreground; }


    /** @returns FeaturePosition */
    inline FeaturePosition featurePosition() const { return _featurePosition; }
    /** @returns the constant font */
    inline const Font& font() const { return _font; }
    /** @returns the font */
    inline Font& font() { return _font; }
    /** @returns true if the Element is Animated */
    inline bool animating() const { return _animating; }
    /** @returns true if State-Changes are animated */
    inline bool animatedStateChanges() const { return _animatedStateChanges; }


    ///////////////////////////////////////////////////////////////
    /// Retrieve Values for the Saved Values inside the States. ///
    ///////////////////////////////////////////////////////////////
    /** @returns left borderWidth @param state the State to retrieve from */
    inline float borderLeft(OrxGui::State state) const { return _style[state]._borderLeft; }
    /** @returns right borderWidth @param state the State to retrieve from */
    inline float borderRight(OrxGui::State state) const { return _style[state]._borderRight; }
    /** @returns top borderWidth @param state the State to retrieve from */
    inline float borderTop(OrxGui::State state) const { return _style[state]._borderTop; }
    /** @returns bottom borderWidth @param state the State to retrieve from */
    inline float borderBottom(OrxGui::State state) const { return _style[state]._borderBottom; }

    /** @returns textSize @param state the State to retrieve from */
    inline float textSize(OrxGui::State state) const { return _style[state]._textSize; }
    /** @returns the Background Color @param state the State to retrieve from */
    inline const Color& backgroundColor(OrxGui::State state) const { return _style[state]._background.diffuseColor(); }
    /** @returns the Background Material. @param state the state to retrieve from */
    inline const Material& background(OrxGui::State state) const { return _style[state]._background; }
    /** @returns background Texture. @param state the State to retrieve from */
    inline const Texture& backgroundTexture(OrxGui::State state) const { return _style[state]._background.diffuseTexture(); }
    /** @returns the foreground Color @param state the State to retrieve from */
    inline const Color& foregroundColor(OrxGui::State state) const { return _style[state]._foreground.diffuseColor(); }
    /** @returns the ForeGroung Material. @param state the state to retrieve from */
    inline const Material& foreground(OrxGui::State state) const { return _style[state]._foreground; }


    /// SETUP
    void resetStyle();
    void loadParams(const TiXmlElement* root);

    void setBorderLeft(float value);
    void setBorderLeft(float value, OrxGui::State state);
    void setBorderLeftS(float value, const std::string& stateName);

    void setBorderRight(float value);
    void setBorderRight(float value, OrxGui::State state);
    void setBorderRightS(float value, const std::string& stateName);

    void setBorderTop(float value);
    void setBorderTop(float value, OrxGui::State state);
    void setBorderTopS(float value, const std::string& stateName);

    void setBorderBottom(float value);
    void setBorderBottom(float value, OrxGui::State state);
    void setBorderBottomS(float value, const std::string& stateName);

    void setTextSize(float value);
    void setTextSize(float value, OrxGui::State state);
    void setTextSizeS(float value, const std::string& stateName);

    void setBackgroundColor(const Color& color);
    void setBackgroundColor(const Color& color, OrxGui::State state);
    void setBackgroundColorS(float r, float g, float b, float a, const std::string& stateName);

    void setBackgroundTexture(const Texture& texture);
    void setBackgroundTexture(const std::string& textureName);
    void setBackgroundTexture(const Texture& texture, OrxGui::State state);
    void setBackgroundTexture(const std::string& textureName, const std::string& stateName);

    void setForegroundColor(const Color& color);
    void setForegroundColor(const Color& color, OrxGui::State state);
    void setForegroundColorS(float r, float g, float b, float a, const std::string& stateName);

    void setForegroundTexture(const Texture& texture);
    void setForegroundTexture(const std::string& textureName);
    void setForegroundTexture(const Texture& texture, OrxGui::State state);
    void setForegroundTexture(const std::string& textureName, const std::string& stateName);

    void loadBackgroundMaterial(const Material& material);
    void loadBackgroundMaterial(const Material& material, OrxGui::State state);
    void loadBackgroundMaterial(const TiXmlElement* element);
    void loadBackgroundMaterial(const TiXmlElement* element, OrxGui::State state);
    void loadBackgroundMaterialS(const TiXmlElement* element, const std::string& stateName);

    void loadForegroundMaterial(const Material& material);
    void loadForegroundMaterial(const Material& material, OrxGui::State state);
    void loadForegroundMaterial(const TiXmlElement* element, OrxGui::State state);
    void loadForegroundMaterialS(const TiXmlElement* element, const std::string& stateName);

    void setFeaturePosition(FeaturePosition featurePosition);
    void setFeaturePositionS(const std::string& featurePosition);

    void setFont(const Font& font);
    void setFont(const std::string& fontName, unsigned int renderSize);

    void setAnimatedStateChanges(bool animated);
    void switchState(OrxGui::State state);
    ///////////////////////////////////////////////////////////////////////////////////


    inline void drawRect(const Rect2D& rect) const
    {
      glBegin(GL_QUADS);
      glTexCoord2i(0,0); glVertex2d(rect.left(), rect.top());
      glTexCoord2i(0,1); glVertex2d(rect.left(), rect.bottom());
      glTexCoord2i(1,1); glVertex2d(rect.right(), rect.bottom());
      glTexCoord2i(1,0); glVertex2d(rect.right(), rect.top());
      glEnd();
    }


    virtual void update() {};
    virtual void tick(float dt);
    virtual void draw() const;

    /** @param the Event to process. @returns true if the Event has been consumed*/
    virtual bool processEvent(const Event& event) { return false; };

    bool getState(const std::string& stateName, OrxGui::State* state);

  protected:
    /** @param focusable sets if the Widget should be focusable */
    void setFocusable(bool focusable = true) { this->_focusable = focusable; };
    /** @param selectable true if the widget should be selectable */
    void setSelectable(bool selectable) { this->_selectable = selectable; }
    /** @param focusable true if the widget should be focusable */
    void setClickable(bool clickable = true) { this->_clickable = clickable; };


    /// RENDERING
    inline void beginDraw() const { glPushMatrix(); glTranslatef(this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0); };
    inline void endDraw() const { glPopMatrix(); };


    /// LOOKS
    virtual void resize();

    virtual void hiding() {};
    virtual void showing() {};

    virtual void updateFrontColor() {};


    /// EVENTS
    // mouse clicking
    virtual void clicking(const Vector2D& pos);
    virtual void releasing(const Vector2D& pos, bool focused);
    // mouse focusing
    virtual void receivedFocus();
    virtual void removedFocus();
    // selecting either with the mouse by clicking, or by the keybord traversing to it.
    virtual void selecting();
    virtual void unselecting();
    // destroying the Widget.
    virtual void destroying();


    virtual void debug(unsigned int level) const;

  private:
    void init();

  private:
    static GLGuiWidget*            _selected;         //!< The currently selected Widget.
    static GLGuiWidget*            _mouseFocused;     //!< The currently Focused Widget (mouse-focus).
    static GLGuiWidget*            _inputGrabber;     //!< The Widget that grabs input (keyboard-focus).


    /// WIDGET
    GLGuiWidget*                   _parent;           //!< The parent of this Widget.

    /// LOOKS
    Rect2D                         _backRect;
    Vector2D                       _minSize;


    /// EVENTS
    OrxGui::State                  _state;
    bool                           _pushed;

    bool                           _focusable;        //!< If this widget can receive focus.
    bool                           _clickable;        //!< if this widget can be clicked upon.
    bool                           _selectable;       //!< If this widget can be selected.




    /// STYLE - Variables.
    typedef struct
    {
      float                        _borderLeft;           //!< The Distance to the left Border of the widget, before any internal Element starts.
      float                        _borderRight;          //!< The Distance to the right Border of the widget, before any internal Element starts.
      float                        _borderTop;            //!< The Distance to the top Border of the widget, before any internal Element starts
      float                        _borderBottom;         //!< The Distance to the bottom Border of the widget, before any internal Element starts

      float                        _textSize;             //!< The TextSize of the Widget.

      Material                     _background;           //!< The Background Material of the Widget.

      Material                     _foreground;           //!< The foreground Material of the Widget.
    }
    StatedStyle;


    StatedStyle                    _style[GLGUI_STATE_COUNT]; //!< Styles configured for different States

    FeaturePosition                _featurePosition;      //!< The Position a Feature will be layed at (checkbox(box), slider(text),...)
    Font                           _font;                 //!< The Font used in the current Widget.


    /// ANIMATION STUFF:
    bool                           _animatedStateChanges; //!< If the Transitions between States are Animated automatically.

    bool                           _animating;            //!< If the Widget is animated at the moment (Texture might be an AnimatedTexture.)
    float                          _animationCycle;
    float                          _animationDuration;
    StatedStyle                    _currentStyle;

  };
}
#endif /* _GLGUI_WIDGET_H */
