/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SHELL

#include "shell_command.h"
#include "shell_command_class.h"

#include "compiler.h"
#include "debug.h"
#include "class_list.h"

#include "key_names.h"

namespace OrxShell
{
  SHELL_COMMAND(debug, ShellCommandClass, help);


  /**
   * @brief constructs and registers a new Command
   * @param commandName the name of the Command
   * @param className the name of the class to apply this command to
   * @param paramCount the count of parameters this command takes
   */
  ShellCommand::ShellCommand(const std::string& commandName, const std::string& className, Executor* executor)
  {
    this->setClassID(CL_SHELL_COMMAND, "ShellCommand");
    PRINTF(4)("create shellcommand '%s' for class '%s'\n", commandName.c_str(), className.c_str());
    this->setName(commandName);

    // copy the executor:
    this->executor = executor;
    this->executor->setName(commandName);

    for (unsigned int i = 0; i < this->executor->getParamCount(); i++)
      this->completors.push_back(new CompletorDefault(&this->executor->getDefaultValue(i)));
    this->alias = NULL;

    //  this->classID = classID;
    this->shellClass = ShellCommandClass::acquireCommandClass(className);
    assert (this->shellClass != NULL);
    this->shellClass->registerCommand(this);
  }

  /**
   * @brief deconstructs a ShellCommand
   */
  ShellCommand::~ShellCommand()
  {
    this->shellClass->unregisterCommand(this);
    if (this->alias != NULL)
      delete this->alias;
    while (!this->completors.empty())
    {
      delete this->completors.back();
      this->completors.pop_back();
    }
    delete this->executor;
  }

  /**
   * @brief registers a new ShellCommand
   */
  ShellCommand* ShellCommand::registerCommand(const std::string& commandName, const std::string& className, Executor* executor)
  {
    if (ShellCommand::exists(commandName, className))
    {
      delete executor;
      return NULL;
    }
    else
      return new ShellCommand(commandName, className, executor);

  }

  /**
   * @brief unregister an existing commandName
   * @param className the name of the Class the command belongs to.
   * @param commandName the name of the command itself
   */
  void ShellCommand::unregisterCommand(const std::string& commandName, const std::string& className)
  {

    ShellCommandClass* cmdClass = ShellCommandClass::acquireCommandClass(className);
    if (cmdClass != NULL)
    {
      CmdList::iterator cmd;
      for (cmd = cmdClass->commandList.begin(); cmd != cmdClass->commandList.end(); cmd++)
        if (commandName == (*cmd)->getName())
        {
          delete (*cmd);
          break;
        }
    }
  }

  /**
   * @brief gets a command if it has already been registered.
   * @param commandName the name of the Command
   * @param cmdClass the CommandClass of the Class the command is in.
   * @returns The Registered Command, or NULL if it does not exist.
   */
  const ShellCommand* const ShellCommand::getCommand(const std::string& commandName, const ShellCommandClass* cmdClass)
  {
    assert(cmdClass != NULL);

    CmdList::const_iterator elem;
    for (unsigned int i = 0; i < cmdClass->commandList.size(); i++)
    {
      if (commandName == cmdClass->commandList[i]->getName())
        return (cmdClass->commandList[i]);
    }
    return NULL;
  }


  /**
   * @brief gets a command if it has already been registered.
   * @param commandName the name of the Command
   * @param className the name of the Class the command is in.
   * @returns The Registered Command, or NULL if it does not exist.
   */
  const ShellCommand* const ShellCommand::getCommand(const std::string& commandName, const std::string& className)
  {
    const ShellCommandClass* checkClass = ShellCommandClass::getCommandClass(className);
    if (likely(checkClass != NULL))
      return ShellCommand::getCommand(commandName, checkClass);
    else
      return NULL;
  }

  /**
   * @brief takes out an InputLine, searching for a Command.
   * @see const ShellCommand* const ShellCommand::getCommandFromInput(const SubString& strings)
   * @param inputLine: the Input to analyse.
   * @param paramBegin: The begin of the Splitted SubStrings entry of the Parameters section.
   * @returns: The ShellCommand if found.
   */
  const ShellCommand* const ShellCommand::getCommandFromInput(const std::string& inputLine, unsigned int& paramBegin, std::vector<BaseObject*>* boList)
  {
    return ShellCommand::getCommandFromInput(SubString(inputLine, SubString::WhiteSpaces), paramBegin, boList);
  }

  /**
   * @brief takes out an InputLine, searching for a Command.
   * @param strings: the Input to analyse.
   * @param paramBegin: The begin of the Splitted SubStrings entry of the Parameters section.
   * @returns: The ShellCommand if found.
   */
  const ShellCommand* const ShellCommand::getCommandFromInput(const SubString& strings, unsigned int& paramBegin, std::vector<BaseObject*>* boList)
  {
    // no input, no Command.
    if (strings.size() == 0)
    {
      paramBegin = 0;
      return NULL;
    }

    // CHECK FOR ALIAS
    std::vector<ShellCommandAlias*>::const_iterator alias;
    for (alias = ShellCommandAlias::getAliases().begin(); alias != ShellCommandAlias::getAliases().end(); alias++ )
    {
      if (strings[0] == (*alias)->getName())
      {
        assert ((*alias)->getCommand() != NULL && (*alias)->getCommand()->shellClass != NULL);
        // Search for Objects.
        if (strings.size() == 1)
        {
          if (fillObjectList("", (*alias)->getCommand(), boList))
            ;
        }
        else
        {
          if (!fillObjectList(strings[1], (*alias)->getCommand(), boList))
            fillObjectList("", (*alias)->getCommand(), boList);
        }
        paramBegin = 1;
        return (*alias)->getCommand();
      }
    }

    // CHECK FOR COMMAND_CLASS
    const ShellCommandClass* cmdClass = ShellCommandClass::getCommandClass(strings[0]);
    if (cmdClass != NULL)
    {
      const ShellCommand* retCmd = NULL;
      // Function/Command right after Class
      if (strings.size() >= 1)
      {
        // Search for Objects.
        retCmd = ShellCommand::getCommand((strings.size() > 1) ? strings[1] : "", cmdClass);
        if (retCmd != NULL)
        {
          paramBegin = 2;
          fillObjectList("", retCmd, boList);
          return retCmd;
        }
      }
      // Function/Command after Class and 'Object'
      if (retCmd == NULL && strings.size() >= 2)
      {
        retCmd = ShellCommand::getCommand((strings.size() > 2) ? strings[2] : "", cmdClass);
        if (retCmd != NULL)
        {
          paramBegin = 3;
          fillObjectList(strings[1], retCmd, boList);
          return retCmd;
        }
      }
      if (retCmd != NULL) // check for the paramBegin.
        return retCmd;
    }
    // Nothing usefull found at all.
    paramBegin = 0;
    return NULL;
  }

  /**
   * @brief fills the ObjectList boList with Objects that can be reffered to by cmd.
   * @param objectDescriptor: the ObjectName (beginning, full name or empty) to fill the List with
   * @param cmd: The Command to complete Objects for.
   * @param boList: The List of BaseObject's that will be filled with found entries.
   * @returns: true if more than one Entry was fond, else (false , or if boList is NULL).
   */
  bool ShellCommand::fillObjectList(const std::string& objectDescriptor, const ShellCommand* cmd, std::vector<BaseObject*>* boList)
  {
    assert (cmd != NULL && cmd->shellClass != NULL);
    if(boList == NULL)
      return false;

    const std::list<BaseObject*>* objectList = ClassList::getList(cmd->shellClass->getName());
    if (objectList != NULL)
    {
      std::list<BaseObject*>::const_iterator bo;

      // No Description given (only for speedup)
      if (objectDescriptor.empty())
      {
        for (bo = objectList->begin(); bo != objectList->end(); bo++)
          boList->push_back(*bo);
      }
      // some description
      else
      {
        for (bo = objectList->begin(); bo != objectList->end(); bo++)
          if (!nocaseCmp(objectDescriptor, (*bo)->getName(), objectDescriptor.size()))
            boList->push_back(*bo);
      }
    }
    return !boList->empty();
  }

  /**
   * @brief checks if a command has already been registered.
   * @param commandName the name of the Command
   * @param className the name of the Class the command should apply to.
   * @returns true, if the command is registered/false otherwise
   *
   * This is used internally, to see, if we have multiple command subscriptions.
   * This is checked in the registerCommand-function.
   */
  bool ShellCommand::exists(const std::string& commandName, const std::string& className)
  {
    return (ShellCommand::getCommand(commandName, className) != NULL);
  }


  /**
   * @brief executes commands
   * @param executionString the string containing the following input
   * ClassName [ObjectName] functionName [parameter1[,parameter2[,...]]]
   * @return true on success, false otherwise.
   */
  bool ShellCommand::execute(const std::string& executionString)
  {
    SubString inputSplits(executionString, SubString::WhiteSpacesWithComma);

    // if we do not have any input return
    if (inputSplits.empty())
      return false;

    unsigned int paramBegin;
    const ShellCommand* sc = NULL;
    std::vector<BaseObject*> boList;
    sc = getCommandFromInput(inputSplits, paramBegin, &boList);
    if (sc != NULL)
    {
      for(std::vector<BaseObject*>::const_iterator bo = boList.begin(); bo != boList.end(); ++bo)
      {
        PRINT(0)("Command '%s' on '%s::%s'\n", sc->getCName(), (*bo)->getClassCName(), (*bo)->getCName());
        (*sc->executor)((*bo), inputSplits.subSet(paramBegin));
      }
      return true;
    }
    return false;
  }


  /**
   * @brief lets a command be described
   * @param description the description of the Given command
   */
  ShellCommand* ShellCommand::describe(const std::string& description)
  {
    if (this == NULL)
      return NULL;
    this->description = description;
    return this;
  }

  /**
   * @brief adds an Alias to this Command
   * @param alias the name of the Alias to set
   * @returns itself
   */
  ShellCommand* ShellCommand::setAlias(const std::string& alias)
  {
    if (this == NULL)
      return NULL;

    if (this->alias != NULL)
    {
      PRINTF(2)("not more than one Alias allowed for functions (%s::%s)\n", this->getCName(), this->shellClass->getCName());
    }
    else
    {
      ShellCommandAlias* aliasCMD = new ShellCommandAlias(alias, this);
      this->alias = aliasCMD;
    }
    return this;
  }

  /**
   * @brief set the default values of the executor
   * @param value0 the first default value
   * @param value1 the second default value
   * @param value2 the third default value
   * @param value3 the fourth default value
   * @param value4 the fifth default value
   */
  ShellCommand* ShellCommand::defaultValues(const MultiType& value0, const MultiType& value1,
      const MultiType& value2, const MultiType& value3,
      const MultiType& value4)
  {
    if (this == NULL || this->executor == NULL)
      return NULL;

    this->executor->defaultValues(value0, value1, value2, value3, value4);

    return this;
  }

  ShellCommand* ShellCommand::completionPlugin(unsigned int parameter, const CompletorPlugin& completorPlugin)
  {
    if (this == NULL || this->executor == NULL)
      return NULL;

    if (parameter >= this->executor->getParamCount())
    {
      PRINTF(1)("Parameter %d not inside of valid ParameterCount %d of Command %s::%s\n",
                parameter, this->executor->getParamCount(), this->getCName(), this->shellClass->getCName());
    }
    else
    {
//       if(this->completors[parameter] == NULL)
//       delete this->completors[parameter];
//       this->completors[parameter] = completorPlugin.clone();
    }
    return this;
  }

  /**
   * @brief prints a Help string from this Command
   */
  void ShellCommand::help() const
  {
    PRINT(0)("%s ", this->getCName());
  }

  /**
   * @brief converts a Parameter to a String
   * @param parameter the Parameter we have.
   * @returns the Name of the Parameter at Hand
   */
  const std::string& ShellCommand::paramToString(long parameter)
  {
    return MultiType::MultiTypeToString((MT_Type)parameter);
  }



  ///////////
  // ALIAS //
  ///////////

  /**
   * @param aliasName the name of the Alias
   * @param command the Command, to associate this alias with
   */
  ShellCommandAlias::ShellCommandAlias(const std::string& aliasName, ShellCommand* command)
  {
    this->aliasName = aliasName;
    this->command = command;
    ShellCommandAlias::aliasList.push_back(this);
  };

  ShellCommandAlias::~ShellCommandAlias()
  {
    std::vector<ShellCommandAlias*>::iterator delA = std::find(aliasList.begin(), aliasList.end(), this);
    if (delA != aliasList.end())
      ShellCommandAlias::aliasList.push_back(this);

  }

  std::vector<ShellCommandAlias*> ShellCommandAlias::aliasList;
  /**
  * @brief collects the Aliases registered to the ShellCommands
  * @param stringList a List to paste the Aliases into.
  * @returns true on success, false otherwise
   */

  bool ShellCommandAlias::getCommandListOfAlias(std::list<std::string>& stringList)
  {
    std::vector<ShellCommandAlias*>::iterator alias;
    for (alias = ShellCommandAlias::aliasList.begin(); alias != ShellCommandAlias::aliasList.end(); alias++)
      stringList.push_back((*alias)->getName());
    return true;
  }


}
