/*!
 * @file executor_functional.h
 * Definition of an Executor
 */

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/


#ifndef __EXECUTOR_FUNCTIONAL_H_
#define __EXECUTOR_FUNCTIONAL_H_

#include "executor.h"

template<typename type> MT_Type ExecutorParamType() { return MT_EXT1; };
template<> MT_Type ExecutorParamType<bool>();
template<> MT_Type ExecutorParamType<int>();
template<> MT_Type ExecutorParamType<unsigned int>();
template<> MT_Type ExecutorParamType<float>();
template<> MT_Type ExecutorParamType<char>();
template<> MT_Type ExecutorParamType<const std::string&>();

template<typename type> type fromString(const std::string& input, type defaultValue) { return defaultValue; };
template<> bool fromString<bool>(const std::string& input, bool defaultValue);
template<> int fromString<int>(const std::string& input, int defaultValue);
template<> unsigned int fromString<unsigned int>(const std::string& input, unsigned int defaultValue);
template<> float fromString<float>(const std::string& input, float defaultValue);
template<> char fromString<char>(const std::string& input, char defaultValue);
template<> const std::string& fromString<const std::string&>(const std::string& input, const std::string& defaultValue);

template<typename type> type fromMulti(const MultiType& multi) { /* return defaultValue; */ };
template<> bool fromMulti<bool>(const MultiType& multi);
template<> int fromMulti<int>(const MultiType& multi);
template<> unsigned int fromMulti<unsigned int>(const MultiType& multi);
template<> float fromMulti<float>(const MultiType& multi);
template<> char fromMulti<char>(const MultiType& multi);
template<> const std::string& fromMulti<const std::string&>(const MultiType& multi);


template<typename type> type getDefault(const MultiType* const defaultValues, unsigned int i) { return (type)0; };
template<> bool getDefault<bool>(const MultiType* const defaultValues, unsigned int i);
template<> int getDefault<int>(const MultiType* const defaultValues, unsigned int i);
template<> unsigned int getDefault<unsigned int>(const MultiType* const defaultValues, unsigned int i);
template<> float getDefault<float>(const MultiType* const defaultValues, unsigned int i);
template<> char getDefault<char>(const MultiType* const defaultValues, unsigned int i);
template<> const std::string& getDefault<const std::string&>(const MultiType* const defaultValues, unsigned int i);

#endif /* __EXECUTOR_FUNCTIONAL_H_ */

//! if Functional is constant calling
#define __EXECUTOR_FUNCTIONAL_CONST
//! The Name to be attached to the functional (for normal, static, and const modes)
#define __EXECUTOR_FUNCTIONAL_NAME(ParamCount)   Executor##ParamCount##Params
//! The Execution-mode (either static or objective)
#define __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC      dynamic_cast<T*>(object)->*(functionPointer)
//! The Function-Pointer, and how to save it internally.
#define __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER   T::*functionPointer

#ifdef EXECUTOR_FUNCTIONAL_USE_CONST
 #undef __EXECUTOR_FUNCTIONAL_CONST
 #define __EXECUTOR_FUNCTIONAL_CONST const
 #undef __EXECUTOR_FUNCTIONAL_NAME
 #define __EXECUTOR_FUNCTIONAL_NAME(ParamCount) Executor##ParamCount##Params_const
#endif

#ifdef EXECUTOR_FUNCTIONAL_USE_STATIC
 #ifdef EXECUTOR_FUNCTIONAL_USE_CONST
  #error you obviously do not know what you are doing !! ask the bensch
 #endif /* EXECUTOR_FUNCTIONAL_USE_CONST */

#undef __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC
 #define __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC       functionPointer
 #undef __EXECUTOR_FUNCTIONAL_NAME
 #define __EXECUTOR_FUNCTIONAL_NAME(ParamCount)    Executor##ParamCount##Params_static
 #undef __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER
 #define __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER    *functionPointer

#endif /* EXECUTOR_FUNCTIONAL_USE_STATIC */

///////////
//// 0 ////
///////////
//! @brief ExecutorClass, that can execute Functions without any parameters.
template<class T> class __EXECUTOR_FUNCTIONAL_NAME(0) : public Executor
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(0) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST )
      : Executor()
  {
    this->functorType = Executor_Objective;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)();
  };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseObject* object, int& count, void* values) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)();
  }

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(0)<T>(this->functionPointer);
  };
};



///////////
//// 1 ////
///////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename type0> class __EXECUTOR_FUNCTIONAL_NAME(1) : public Executor
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(1) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor(ExecutorParamType<type0>())
  {
    this->functorType = Executor_Objective;
    this->functionPointer = functionPointer;
  };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseObject* object, int& count, void* values) const
  {
    const MultiType* mt = (const MultiType*)values;
    return (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
             fromMulti<type0>((count > 0)? mt[0] : this->defaultValue[0]) );
  }


  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      (sub.size() > 0) ? fromString<type0>(sub[0], getDefault<type0>(this->defaultValue, 0)) : (fromMulti<type0>(this->defaultValue[0])));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(1)<T, type0>(this->functionPointer);
  };
};

///////////
//// 2 ////
///////////
//! @brief ExecutorClass, that can execute Functions with two parameters.
template<class T, typename type0, typename type1> class __EXECUTOR_FUNCTIONAL_NAME(2) : public Executor
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(2) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor(ExecutorParamType<type0>(), ExecutorParamType<type1>())
  {
    this->functorType = Executor_Objective;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      (sub.size() > 0) ? fromString<type0>(sub[0], getDefault<type0>(this->defaultValue, 0)) : (fromMulti<type0>(this->defaultValue[0])),
      (sub.size() > 1) ? fromString<type1>(sub[1], getDefault<type1>(this->defaultValue, 1)) : (fromMulti<type1>(this->defaultValue[1])));
  };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseObject* object, int& count, void* values) const
  {
    const MultiType* mt = (const MultiType*)values;
    return (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
             fromMulti<type0>((count > 0) ? mt[0] : this->defaultValue[0]),
             fromMulti<type1>((count > 1) ? mt[1] : this->defaultValue[1]) );
  }

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(2)<T, type0, type1>(this->functionPointer);
  };
};


///////////
//// 3 ////
///////////
//! @brief ExecutorClass, that can execute Functions with three parameters.
template<class T, typename type0, typename type1, typename type2> class __EXECUTOR_FUNCTIONAL_NAME(3) : public Executor
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(3) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor(ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>())
  {
    this->functorType = Executor_Objective;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      (sub.size() > 0) ? fromString<type0>(sub[0], getDefault<type0>(this->defaultValue, 0)) : (fromMulti<type0>(this->defaultValue[0])),
      (sub.size() > 1) ? fromString<type1>(sub[1], getDefault<type1>(this->defaultValue, 1)) : (fromMulti<type1>(this->defaultValue[1])),
      (sub.size() > 2) ? fromString<type2>(sub[2], getDefault<type2>(this->defaultValue, 2)) : (fromMulti<type2>(this->defaultValue[2])));
  };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseObject* object, int& count, void* values) const
  {
    const MultiType* mt = (const MultiType*)values;
    return (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
             fromMulti<type0>((count > 0) ? mt[0] : this->defaultValue[0]),
             fromMulti<type1>((count > 1) ? mt[1] : this->defaultValue[1]),
             fromMulti<type2>((count > 2) ? mt[2] : this->defaultValue[2]) );
  }

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(3)<T, type0, type1, type2>(this->functionPointer);
  };
};



///////////
//// 4 ////
///////////
//! @brief ExecutorClass, that can execute Functions with four parameters.
template<class T, typename type0, typename type1, typename type2, typename type3> class __EXECUTOR_FUNCTIONAL_NAME(4) : public Executor
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(4) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor(ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>())
  {
    this->functorType = Executor_Objective;
    this->functionPointer = functionPointer;
  };

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      (sub.size() > 0) ? fromString<type0>(sub[0], getDefault<type0>(this->defaultValue, 0)) : (fromMulti<type0>(this->defaultValue[0])),
      (sub.size() > 1) ? fromString<type1>(sub[1], getDefault<type1>(this->defaultValue, 1)) : (fromMulti<type1>(this->defaultValue[1])),
      (sub.size() > 2) ? fromString<type2>(sub[2], getDefault<type2>(this->defaultValue, 2)) : (fromMulti<type2>(this->defaultValue[2])),
      (sub.size() > 3) ? fromString<type3>(sub[3], getDefault<type3>(this->defaultValue, 3)) : (fromMulti<type3>(this->defaultValue[3])));
  };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseObject* object, int& count, void* values) const
  {
    const MultiType* mt = (const MultiType*)values;
    return (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
             fromMulti<type0>((count > 0) ? mt[0] : this->defaultValue[0]),
             fromMulti<type1>((count > 1) ? mt[1] : this->defaultValue[1]),
             fromMulti<type2>((count > 2) ? mt[2] : this->defaultValue[2]),
             fromMulti<type3>((count > 3) ? mt[3] : this->defaultValue[3]) );
  }

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(4)<T, type0, type1, type2, type3>(this->functionPointer);
  };
};



///////////
//// 5 ////
///////////
//! @brief ExecutorClass, that can execute Functions with five parameters.
template<class T, typename type0, typename type1, typename type2, typename type3, typename type4> class __EXECUTOR_FUNCTIONAL_NAME(5) : public Executor
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(5) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor(ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>(), ExecutorParamType<type4>())
  {
    this->functorType = Executor_Objective;
    this->functionPointer = functionPointer;
  };

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      (sub.size() > 0) ? fromString<type0>(sub[0], getDefault<type0>(this->defaultValue, 0)) : (fromMulti<type0>(this->defaultValue[0])),
      (sub.size() > 1) ? fromString<type1>(sub[1], getDefault<type1>(this->defaultValue, 1)) : (fromMulti<type1>(this->defaultValue[1])),
      (sub.size() > 2) ? fromString<type2>(sub[2], getDefault<type2>(this->defaultValue, 2)) : (fromMulti<type2>(this->defaultValue[2])),
      (sub.size() > 3) ? fromString<type3>(sub[3], getDefault<type3>(this->defaultValue, 3)) : (fromMulti<type3>(this->defaultValue[3])),
      (sub.size() > 4) ? fromString<type4>(sub[4], getDefault<type4>(this->defaultValue, 4)) : (fromMulti<type4>(this->defaultValue[4])));
  };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseObject* object, int& count, void* values) const
  {
    const MultiType* mt = (const MultiType*)values;
    return (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
             fromMulti<type0>((count > 0) ? mt[0] : this->defaultValue[0]),
             fromMulti<type1>((count > 1) ? mt[1] : this->defaultValue[1]),
             fromMulti<type2>((count > 2) ? mt[2] : this->defaultValue[2]),
             fromMulti<type3>((count > 3) ? mt[3] : this->defaultValue[3]),
             fromMulti<type4>((count > 4) ? mt[4] : this->defaultValue[4]) );
  }

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(5)<T, type0, type1, type2, type3, type4>(this->functionPointer);
  };
};



/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 */
#define EXECUTOR_FUNCTIONAL_CREATOR0() \
template<class T> Executor* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(0)<T>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR1(type0) \
template<class T> Executor* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0##_TYPE) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(1)<T, type0##_TYPE>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR2(type0, type1) \
template<class T> Executor* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0##_TYPE, type1##_TYPE) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(2)<T, type0##_TYPE, type1##_TYPE>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 * @param type2 for internal usage: the third Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR3(type0, type1, type2) \
template<class T> Executor* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0##_TYPE, type1##_TYPE, type2##_TYPE) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(3)<T, type0##_TYPE, type1##_TYPE, type2##_TYPE>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 * @param type2 for internal usage: the third Argument
 * @param type3 for internal usage: the fourth Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR4(type0, type1, type2, type3) \
template<class T> Executor* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0##_TYPE, type1##_TYPE, type2##_TYPE, type3##_TYPE) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(4)<T, type0##_TYPE, type1##_TYPE, type2##_TYPE, type3##_TYPE>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 * @param type2 for internal usage: the third Argument
 * @param type3 for internal usage: the fourth Argument
 * @param type4 for internal usage: the fifth Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR5(type0, type1, type2, type3, type4) \
template<class T> Executor* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0##_TYPE, type1##_TYPE, type2##_TYPE, type3##_TYPE, type4##_TYPE) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
    return new __EXECUTOR_FUNCTIONAL_NAME(5)<T, type0##_TYPE, type1##_TYPE, type2##_TYPE, type3##_TYPE, type4##_TYPE>(functionPointer); \
}


/**
 * Creates the FunctionCallers
 */
#define FUNCTOR_LIST(x) EXECUTOR_FUNCTIONAL_CREATOR ## x
#include "functor_list.h"
#undef FUNCTOR_LIST



#undef __EXECUTOR_FUNCTIONAL_CONST
#undef __EXECUTOR_FUNCTIONAL_NAME
#undef __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC
#undef __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER

#ifdef EXECUTOR_FUNCTIONAL_USE_CONST
 #undef EXECUTOR_FUNCTIONAL_USE_CONST
#endif
#ifdef EXECUTOR_FUNCTIONAL_USE_STATIC
 #undef EXECUTOR_FUNCTIONAL_USE_STATIC
#endif
