/*!
 * @file network_stream.h
 *  implementation of a network pipe. This node handles the handshake of two network nodes and exchanges informationas
 * vial to the connection setup.
 *
 * the local node only writes in the localState variable. the remoteState variable will be written from the remote network node
 * so don't write into it!
 */

#ifndef _HANDSHAKE
#define _HANDSHAKE

#include "base_object.h"
#include "ip.h"
#include "synchronizeable.h"

//!< a struct to save the handshakes to
struct HandshakeState
{
  int           orxId;                         //!< orxonox id
  int           version;                       //!< network protocol version

  int           networkManagerId;              //!< unique id of the network manager
  int           messageManagerId;              //!< unique id of the message manager
  int           hostId;                        //!< host id
  int           nodeType;                      //!< type of the network node

  int           completed;                     //!< true if completed
  int           canDel;                        //!< true if marked for deletion

  int           error;                         //!< error number

  std::string   errorString;                   //!< error string

  //additional data
  std::string   preferedNickName;              //!< prefered nick name

  int           redirectProxy;                 //!< true if the client should reconnect to a proxy server (either 1 or 2 )
  IP            proxy1;                        //!< ip address of the first proxy (0.0.0.0 of not available)
  IP            proxy2;                        //!< ip address of the second proxy (0.0.0.0 of not available)
};


//!< the handshake itself with some interface functions
class Handshake : public Synchronizeable
{

  public:
    Handshake( int nodeType, int clientId = 0, int networkGameManagerId = 0, int messageManagerId = 0 );


    /* functions indicating states of the handshake */
    /** @returns true if the handshake is completed */
    inline bool completed(){ return localState.completed != 0 && remoteState.completed != 0; }
    /** @returns true if no error has occured until now */
    inline bool ok(){ return localState.error == 0 && remoteState.error == 0; }
    /** stops the handshake and reject the other side with @param reason: string describing the reason */
    inline void doReject( std::string reason ){ localState.error = 1; localState.errorString = "the server rejected your connection ["+ reason +"]"; }
    /** @returns true if the handshake is finished and the instances can be deleted */
    inline bool canDel(){ return localState.canDel == 1 && remoteState.canDel == 1; }
    /** @returns true if the local state can be removed*/
    inline bool allowDel(){ return localState.canDel == 1; }
    /** marks the handshake to be deleted */
    inline void del(){ localState.canDel = 1; }


    /* the actual informations exchanged in the handshake */
    /** @returns the host id of the remote host */
    inline int  getHostId(){ return remoteState.hostId; }
    /** @returns the unique id of the network game manager*/
    inline int  getNetworkGameManagerId(){ return remoteState.networkManagerId; }
    /** @returns the unique id of the message manager */
    inline int  getMessageManagerId(){ return remoteState.messageManagerId; }
    /** @returns the node type of the remote host */
    inline int getRemoteNodeType() { return this->remoteState.nodeType; }

    /** sets @param nick the prefereded nick name */
    inline void setPreferedNickName( const std::string & nick ){ localState.preferedNickName = nick; }
    /** @returns the prefered nick name */
    inline std::string getPreferedNickName(){ return remoteState.preferedNickName; }

    /** @returns if true the local client should reconnect to a proxy server from the proxy server list */
    inline bool redirect() { return this->remoteState.redirectProxy;}
    /** @param flag: indicating if the client should be redirected */
    inline void setRedirect(bool flag) { this->localState.redirectProxy = flag; }

    /** @param address: the address of the proxy server 1 if any */
    inline void setProxy1Address(IP address) { this->localState.proxy1 = address; }
    /** @returns the address of the proxy server 1 if any */
    inline IP getProxy1Address() { return this->remoteState.proxy1; }
    /** @param address: the address of the proxy server 2 if any */
    inline void setProxy2Address(IP address) { this->localState.proxy2 = address; }
    /** @returns the address of the proxy server 2 if any */
    inline IP getProxy2Address() { return this->remoteState.proxy2; }


    /* variable handler function */
    virtual void varChangeHandler( std::list<int> & id );


  private:
    HandshakeState     localState;                            //!< the local handshake state
    HandshakeState     remoteState;                           //!< the remote handshake state

    int                orxId_handler;                         //!< orxonox id handler
    int                version_handler;                       //!< orxonox version id handler
    int                netManId_handler;                      //!< network manager handler
    int                msgManId_handler;                      //!< message manager handler
    int                hostId_handler;                        //!< host id handler
    int                nodeTypeHandler;                       //!< node type handler

    int                completed_handler;                     //!< handshake completion handler
    int                error_handler;                         //!< handshake error handler
    int                errorString_handler;                   //!< handshake error string handler
    int                candel_id;                             //!< handshake deletion handler
    int                nodeType;                              //!, the type of the network node

};


#endif
