/*!
 * @file network_monitor.h
 *  a class to monitor the network: throughput, network nodes
 */

#ifndef _NETWORK_MONITOR_H
#define _NETWORK_MONITOR_H

#include "base_object.h"
#include "synchronizeable.h"
#include "shared_network_data.h"

#include "network_node.h"

#include <list>

class NetworkStream;
class PeerInfo;


namespace OrxGui { class GLGuiBox; };



//!< a class that monitors the whole network. it tries to gather all informations its able to from the network
class NetworkMonitor : public Synchronizeable
{

  public:
    NetworkMonitor(NetworkStream* networkStream);
    virtual ~NetworkMonitor();

    /* client/masterserver/proxyserver addition/removal interface */
    void addNetworkNode(NetworkNode* node);
    void removeNetworkNode(NetworkNode* node);

    void addNode(PeerInfo* pInfo);
    void addNode(IPaddress ip, int nodeType);
    void addNode(NetworkNode* node) { this->nodeList.push_back(node); }
    void addNode(NetworkNode* node, PeerInfo* pInfo);

    /** adds to @param node a network node @param pInfo a new client */
    inline void addClient(NetworkNode* node, PeerInfo* pInfo) { node->addClient(pInfo); }
    /** adds to @param node a network node @param pInfo a new proxy server */
    inline void addActiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->addActiveProxyServer(pInfo); }
    /** adds to @param node a network node @param pInfo a new proxy server */
    inline void addPassiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->addPassiveProxyServer(pInfo); }
    /** adds to @param node a network node @param pInfo a new master server*/
    inline void addMasterServer(NetworkNode* node, PeerInfo* pInfo) { node->addMasterServer(pInfo); }

    inline void removeClient(NetworkNode* node, PeerInfo* pInfo) { node->removeClient(pInfo); }
    inline void removeActiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->removeActiveProxyServer(pInfo); }
    inline void removePassiveProxyServer(NetworkNode* node, PeerInfo* pInfo) { node->removePassiveProxyServer(pInfo); }
    inline void removeMasterServer(NetworkNode* node, PeerInfo* pInfo) { node->removeMasterServer(pInfo); }

    /* slots admin and info interface */
    /** @returns the total number of players in this game (including all proxy servers etc)*/
    inline int getPlayerNumber() { return this->playerNumber; }

    /** @returns true if there are still free network slots available */
    inline bool gotFreeSlots() { return (this->localNode->getPlayerNumber() < SharedNetworkData::getInstance()->getMaxPlayer())?true:false; }
    /** @param node node to be checked for slots @returns true if there are still free network slots available at this node */
    inline bool gotFreeSlots(NetworkNode* node) { return (node->getPlayerNumber() < SharedNetworkData::getInstance()->getMaxPlayer())?true:false; }
    PeerInfo* getFirstChoiceProxy();
    PeerInfo* getSecondChoiceProxy();
    /** @returns true if the next client should be reconnected to some other proxy server with more connections */
    inline bool reconnectNextClient() { return (this->localNode->getPlayerNumber() >= SharedNetworkData::getInstance()->getMaxPlayer())?true:false; }


    void showGUI();
    void hideGUI();

    void process();
    void debug();


  private:
    NetworkStream*               networkStream;              //!< reference to the one networkstream of the network
    NetworkNode*                 localNode;                  //!< reference to the local node
    std::list<NetworkNode*>      nodeList;                   //!< list of all network nodes

    OrxGui::GLGuiBox*            box;

    int                          playerNumber;                 //!< total number of players in the game
};


#endif
