/*!
 * @file network_stream.h
 *  implementation of a network pipe
 */

#ifndef _HANDSHAKE
#define _HANDSHAKE

#include "base_object.h"
#include "synchronizeable.h"


struct HandshakeState {
  int orxId;                         //!< orxonox id
  int version;                       //!< network protocol version

  int networkManagerId;              //!< unique id of the network manager
  int messageManagerId;              //!< unique id of the message manager
  int hostId;                        //!< host id
  int nodeType;                      //!< type of the network node

  int completed;                     //!< true if completed
  int canDel;                        //!< true if marked for deletion

  int error;                         //!< error number

  std::string errorString;           //!< error string

  //additional data
  std::string preferedNickName;      //!< prefered nick name
};

class Handshake : public Synchronizeable
{

  public:
    Handshake( bool server, int clientId = 0, int networkGameManagerId = 0, int messageManagerId = 0 );


    /* functions indicating states of the handshake */
    /** @returns true if the handshake is completed */
    inline bool completed(){ return localState.completed != 0 && remoteState.completed != 0; }
    /** @returns true if no error has occured until now */
    inline bool ok(){ return localState.error == 0 && remoteState.error == 0; }
    inline void doReject( std::string reason ){ localState.error = 1; localState.errorString = "the server rejected your connection ["+ reason +"]"; }
    /** @returns true if the handshake is finished and the instances can be deleted */
    inline bool canDel(){ return localState.canDel == 1 && remoteState.canDel == 1; }
    /** @returns true if the local state can be removed*/
    inline bool allowDel(){ return localState.canDel == 1; }
    /** marks the handshake to be deleted */
    inline void del(){ localState.canDel = 1; }

    /* the actual informations exchanged in the handshake */
    /** @returns the host id of the remote host */
    inline int  getHostId(){ return remoteState.hostId; }
    /** @returns the unique id of the network game manager*/
    inline int  getNetworkGameManagerId(){ return remoteState.networkManagerId; }
    /** @returns the unique id of the message manager */
    inline int  getMessageManagerId(){ return remoteState.messageManagerId; }
    /** stops the handshake and reject the other side with @param reason: string describing the reason */

    /** sets @param nick the prefereded nick name */
    inline void setPreferedNickName( const std::string & nick ){ localState.preferedNickName = nick; }
    /** @returns the prefered nick name */
    inline std::string getPreferedNickName(){ return remoteState.preferedNickName; }

    /* variable handler function */
    virtual void varChangeHandler( std::list<int> & id );


  private:
    HandshakeState     localState;                            //!< the local handshake state
    HandshakeState     remoteState;                           //!< the remote handshake state

    int                orxId_handler;                         //!< orxonox id handler
    int                version_handler;                       //!< orxonox version id handler
    int                netManId_handler;                      //!< network manager handler
    int                msgManId_handler;                      //!< message manager handler
    int                hostId_handler;                        //!< host id handler
    int                nodeTypeHandler;                       //!< node type handler

    int                completed_handler;                     //!< handshake completion handler
    int                error_handler;                         //!< handshake error handler
    int                errorString_handler;                   //!< handshake error string handler
    int                candel_id;                             //!< handshake deletion handler
    int                nodeType;                              //!, the type of the network node

};


#endif
