/*!
 * @file network_stream.h
 *  implementation of a network pipe
 */

#ifndef _HANDSHAKE
#define _HANDSHAKE

#include "base_object.h"
#include "synchronizeable.h"

#define _ORXONOX_ID 0xF91337A0

#define _ORXONOX_VERSION 1

struct HandshakeState {
  int orxId;
  int version;

  int networkManagerId;
  int messageManagerId;
  int hostId;

  int completed;
  int canDel;

  int error;

  std::string errorString;

  //additional data
  std::string preferedNickName;
};

class Handshake : public Synchronizeable
{

  public:
    Handshake( bool server, int clientId = 0, int networkGameManagerId = 0, int messageManagerId = 0 );


    /* functions indicating states of the handshake */
    /** @returns true if the handshake is completed */
    inline bool completed(){ return localState.completed != 0 && remoteState.completed != 0; }
    /** @returns true if no error has occured until now */
    inline bool ok(){ return localState.error == 0 && remoteState.error == 0; }
    inline void doReject( std::string reason ){ localState.error = 1; localState.errorString = "the server rejected your connection ["+ reason +"]"; }
    /** @returns true if the handshake is finished and the instances can be deleted */
    inline bool canDel(){ return localState.canDel == 1 && remoteState.canDel == 1; }
    /** @returns true if the local state can be removed*/
    inline bool allowDel(){ return localState.canDel == 1; }
    /** marks the handshake to be deleted */
    inline void del(){ localState.canDel = 1; }

    /* the actual informations exchanged in the handshake */
    /** @returns the host id of the remote host */
    inline int  getHostId(){ return remoteState.hostId; }
    /** @returns the unique id of the network game manager*/
    inline int  getNetworkGameManagerId(){ return remoteState.networkManagerId; }
    /** @returns the unique id of the message manager */
    inline int  getMessageManagerId(){ return remoteState.messageManagerId; }
    /** stops the handshake and reject the other side with @param reason: string describing the reason */

    /** sets @param nick the prefereded nick name */
    inline void setPreferedNickName( const std::string & nick ){ localState.preferedNickName = nick; }
    /** @returns the prefered nick name */
    inline std::string getPreferedNickName(){ return remoteState.preferedNickName; }

    /* variable handler function */
    virtual void varChangeHandler( std::list<int> & id );


  private:
    HandshakeState     localState;                            //!< the local handshake state
    HandshakeState     remoteState;                           //!< the remote handshake state

    int                netManId_handler;                      //!< network manager handler
    int                msgManId_handler;                      //!< message manager handler
    int                hostId_handler;
    int                completed_handler;
    int                error_handler;
    int                errorString_handler;
    int                orxId_handler;
    int                version_handler;
    int                candel_id;

};


#endif
