/*!
 * @file network_stream.h
 *  implementation of a network pipe
 */

#ifndef _NETWORK_STREAM
#define _NETWORK_STREAM

#include <vector>
#include <list>
#include <map>

#include "data_stream.h"
#include "server_socket.h"
#include "handshake.h"
#include "connection_monitor.h"
#include "udp_server_socket.h"

class Synchronizeable;
class NetworkSocket;
class ServerSocket;
class NetworkGameManager;

//!< this structure contains informations about the network node
class PeerInfo
{
  public:
    PeerInfo() { clear(); }
    void clear() { userId = 0; isServer = false; socket = NULL; handshake = NULL; lastAckedState = 0; lastRecvedState = 0; connectionMonitor = NULL; }


  public:
    int                 userId;
    bool                isServer;
    NetworkSocket *     socket;
    Handshake *         handshake;
    ConnectionMonitor * connectionMonitor;
    int                 lastAckedState;
    int                 lastRecvedState;
};

typedef std::list<Synchronizeable*>  SynchronizeableList;
typedef std::map<int,PeerInfo>       PeerList;


class NetworkStream : public DataStream
{

  public:
    NetworkStream();
    NetworkStream( std::string host, int port);
    NetworkStream( int port );

    virtual ~NetworkStream();
    void init();

    void createNetworkGameManager();
    void startHandshake();

    void connectSynchronizeable(Synchronizeable& sync);
    void disconnectSynchronizeable(Synchronizeable& sync);

    inline bool isServer() const { return (this->type == NET_SERVER)? true:false; }
    inline bool isActive() const { return this->bActive; }

    inline int getMaxConnections(){ return MAX_CONNECTIONS; }

    virtual void processData();

    inline SynchronizeableList::const_iterator getSyncBegin(){ return synchronizeables.begin(); }
    inline SynchronizeableList::const_iterator getSyncEnd(){ return synchronizeables.end(); }
    int getSyncCount();

    inline bool isUserIdActive( int userID ) { return (peers.find(userID) != peers.end()); }
    inline bool isUserServer( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isServer; }

    void debug();

    inline PeerInfo & getPeerInfo( int userId ) { return peers[userId]; }


  private:
    void updateConnectionList();
    void handleHandshakes();
    void handleUpstream( int tick );
    void handleDownstream(int tick );
    void handleNewClient( int userId );
    void cleanUpOldSyncList();

    void writeToNewDict( byte * data, int length, bool upstream );


  private:
    SynchronizeableList        synchronizeables;
    PeerList                   peers;
    ServerSocket*              serverSocket;
    int                        type;
    bool                       bActive;
    std::list<int>             freeSocketSlots;

    int                        myHostId;

    int                        currentState;                //!< current state id

    NetworkGameManager*        networkGameManager;

    std::map<int,int>          oldSynchronizeables;         //!< used to save recently deleted sync ids to not recreate them

    byte                       buf[UDP_PACKET_SIZE];        //!< used by handleUp/Downstream
    byte                       compBuf[UDP_PACKET_SIZE];    //!< used by handleUp/Downstream

    int                        remainingBytesToWriteToDict; //!< if > 0 NetworkStream will write packets to DATA/dicts/newdict

    int dictServer;
    int dictClient;
};
#endif /* _NETWORK_STREAM */
