/*
  orxonox - the future of 3D-vertical-scrollers

  Copyright (C) 2004 orx

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

### File Specific:
  main-programmer: hdavid, amaechler
*/

#include "fog_effect.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "shell_command.h"
#include "script_class.h"
#include "cloud_effect.h"

// Define shell commands
//SHELL_COMMAND(activate, FogEffect, activateFog);
//SHELL_COMMAND(deactivate, FogEffect, deactivateFog);
SHELL_COMMAND(fadein, FogEffect, fadeInFog);
SHELL_COMMAND(fadeout, FogEffect, fadeOutFog);

using namespace std;

CREATE_SCRIPTABLE_CLASS(FogEffect, CL_FOG_EFFECT,
                        addMethod("fadeIn", ExecutorLua0<FogEffect>(&FogEffect::fadeInFog))
                            ->addMethod("fadeOut", ExecutorLua0<FogEffect>(&FogEffect::fadeOutFog))
                            ->addMethod("activate", ExecutorLua0<FogEffect>(&FogEffect::activate))
                            ->addMethod("deactivate", ExecutorLua0<FogEffect>(&FogEffect::deactivate))
                       );

CREATE_FACTORY(FogEffect, CL_FOG_EFFECT);

/**
 * @brief standard constructor
 */
FogEffect::FogEffect(const TiXmlElement* root) {
    this->setClassID(CL_FOG_EFFECT, "FogEffect");

    // Initialize values
    this->init();

    // Load XML params
    if (root != NULL)
        this->loadParams(root);

    // Activate fog, if chosen to be activated by default
    if (this->fogActivate)
        this->activate();
}

/**
 * @brief standard destructor
 */
FogEffect::~FogEffect() {
    this->deactivate();
}

/**
 * @brief initalizes the fog effect with default values
 */
void FogEffect::init() {
    // default values
  this->fogMode = GL_LINEAR;
  this->fogDensity = 0.005;
  this->fogStart = 0;
  this->fogEnd = 300;
  this->colorVector = Vector(0.6, 0.0, 0.0);

    // init variables
  this->fogFadeInDuration = 0;
  this->fogFadeOutDuration = 0;
  this->fogFadeDensity = 0;
  this->fogFadeEnd = 0;

  this->localTimer = 0;
  this->fogActivate = false;
  this->fogFadeInActivate = false;
  this->fogFadeOutActivate = false;
  
  this->cloudColor = Vector(0.2f, 0.3f, 0.3f);
  this->skyColor = Vector(0.2f, 0.3f, 0.3f);
}

/**
 * @brief loads the fog effect parameters.
 * @param root: the XML-Element to load the data from
 */
void FogEffect::loadParams(const TiXmlElement* root) {
    WeatherEffect::loadParams(root);

    LoadParam(root, "mode", this, FogEffect, setFogMode).describe("fog mode (linear, exponential)");
    LoadParam(root, "density", this, FogEffect, setFogDensity).describe("fog density if exp. fog");
    LoadParam(root, "range", this, FogEffect, setFogRange).describe("fog range: start, end");
    LoadParam(root, "color", this, FogEffect, setFogColor).describe("fog color: r,g,b");
    LoadParam(root, "fadeinduration", this, FogEffect, setFogFadeIn).describe("duration of the fade in");
    LoadParam(root, "fadeoutduration", this, FogEffect, setFogFadeOut).describe("duration of the fade out");
    LoadParam(root, "cloudcolor", this, FogEffect, setCloudColor);
    LoadParam(root, "skycolor", this, FogEffect, setSkyColor);
    
    LOAD_PARAM_START_CYCLE(root, element);
    {
      LoadParam_CYCLE(element, "option", this, FogEffect, setFogOption).describe("sets a fog option: activate");
    }
    LOAD_PARAM_END_CYCLE(element);
}

/**
 * @brief activates the fog effect
 */
void FogEffect::activate() {
    PRINTF(3)( "Activating FogEffect\n");

    // init fogGL
    GLfloat fogColor[4] = { colorVector.x, colorVector.y, colorVector.z, 1.0};
    glFogi(GL_FOG_MODE, this->fogMode);
    glFogfv(GL_FOG_COLOR, fogColor);
    glHint(GL_FOG_HINT, GL_DONT_CARE);
    glFogf(GL_FOG_DENSITY, this->fogDensity);
    glFogf(GL_FOG_START, this->fogStart);
    glFogf(GL_FOG_END, this->fogEnd);

    this->fogActivate = true;

    glEnable(GL_FOG);
    
    // Store cloud- and sky color before the snow
    this->oldCloudColor = CloudEffect::cloudColor;
    this->oldSkyColor   = CloudEffect::skyColor;
    
    // Change the colors
    CloudEffect::changeCloudColor(this->cloudColor, this->fogFadeInDuration);
    CloudEffect::changeSkyColor(this->skyColor, this->fogFadeInDuration);
}

/**
 * @brief deactivates the fog effect
 */
void FogEffect::deactivate() {
    PRINTF(3)("Deactivating FogEffect\n");

    this->fogFadeInActivate = false;
    this->fogFadeOutActivate = false;
    this->fogActivate = false;

    glDisable(GL_FOG);
}

/**
 * @brief draws the fog effect
 */
void FogEffect::draw() const {

    if (!this->fogActivate)
        return;

    // If fog is fading
    if ( this->fogFadeInActivate || this->fogFadeOutActivate ) {
        if ( this->fogMode == GL_LINEAR)
            glFogf(GL_FOG_END, this->fogFadeEnd);
        else
            glFogf(GL_FOG_DENSITY, this->fogFadeDensity);
    }
}

/**
 * @brief ticks the fog effect
 * @param dt: tick float
 */
void FogEffect::tick(float dt) {

    if (!this->fogActivate)
        return;

    // If fog is fading in
    if ( this->fogFadeInActivate ) {
        this->localTimer += dt;

        if ( this->fogMode == GL_LINEAR)
          this->fogFadeEnd = 2000 * (1 - ( this->localTimer / this->fogFadeInDuration )) + this->fogEnd;
        else
            this->fogFadeDensity = ( this->localTimer / this->fogFadeInDuration ) * this->fogDensity;

        if ( this->localTimer >= this->fogFadeInDuration )
            this->fogFadeInActivate = false;
    }

    // If fog is fading out
    if ( this->fogFadeOutActivate ) {
        this->localTimer += dt;

        if ( this->fogMode == GL_LINEAR)
          this->fogFadeEnd = 2000 * ( this->localTimer / this->fogFadeOutDuration ) + this->fogEnd;
        else
            this->fogFadeDensity = 1 - (( this->localTimer / this->fogFadeOutDuration ) * this->fogDensity);

        if ( this->localTimer >= this->fogFadeOutDuration )
            this->deactivate();
    }
}

/**
 * @brief fades the fog in
 */
void FogEffect::fadeInFog() {

    // If Fog is already fading out, stop it
    this->fogFadeOutActivate = false;

    // If Fog is already on, turn it off first
    if (this->fogActivate)
        this->deactivate();

    // If no manual FadeIn value was set, set a default value
    if (!this->fogFadeInDuration > 0)
        this->fogFadeInDuration = 10;

    // Reset local timer
    this->localTimer = 0;

    // Activate Fog
    this->activate();

    // set FogFadeIn activate
    this->fogFadeInActivate = true;
}

/**
 * @brief fades the fog out
 */
void FogEffect::fadeOutFog() {

    // If Fog is already fading in, stop it
    this->fogFadeInActivate = false;

    
    // If Fog is off, turn it on first
    if (!this->fogActivate)
        this->activate();

    // If no manual FadeOut value was set, set a default value
    if (!this->fogFadeOutDuration > 0)
        this->fogFadeOutDuration = 10;

    // set FogFadeOut activate
    this->fogFadeOutActivate = true;

    // Reset local timer
    this->localTimer = 0;

    // Restore the old cloud- and sky color
    CloudEffect::changeCloudColor(this->oldCloudColor, this->fogFadeOutDuration);
    CloudEffect::changeSkyColor(this->oldSkyColor, this->fogFadeOutDuration);
}

