/*!
 * @file bv_tree.h
 *  Definition of a bounding volume tree
 */

#ifndef _OBB_TREE_NODE_H
#define _OBB_TREE_NODE_H


#include "bv_tree_node.h"
#include "plane.h"

class BoundingVolume;
class OBB;
class OBBTree;
class Plane;
class PNode;


//! A class that represents a bounding volume tree
class OBBTreeNode : public BVTreeNode
{


  public:
    OBBTreeNode(const OBBTree& tree, OBBTreeNode* prev, int depth);
    virtual ~OBBTreeNode();

    /**  this function returns the bounding volume of this tree node @return: returns the BV */
    virtual inline const BoundingVolume* getBV() const { return (BoundingVolume*)this->bvElement; }

    virtual void spawnBVTree(const modelInfo& modelInf, const int* triangleIndexes, int length);
    void createBox(Vector start, Vector end);

    virtual void collideWith(BVTreeNode* treeNode, WorldEntity* nodeA, WorldEntity* nodeB);
    virtual void drawBV(int depth, int drawMode, const Vector& color = Vector(1,0,0), bool top = true) const;
    void debug() const;

    /**  gets the id of the current child @return id of the child */
    inline const int getID() { return this->nextID++; }


  private:
    void calculateBoxAxis(OBB& box, const sVec3D* verticesList, unsigned int length);

    void calculateBoxCovariance(OBB& box, const modelInfo& modelInf, const int* triangleIndexes, int length);
    void calculateBoxEigenvectors(OBB& box, const modelInfo& modelInf, const int* triangleIndexes, int length);
    void calculateBoxAxis(OBB& box, const modelInfo& modelInf, const int* triangleIndexes, int length);
    void forkBox(OBB& box);

    void collideWithOBB(OBBTreeNode* treeNode, WorldEntity* nodeA, WorldEntity* nodeB);
    bool overlapTest(OBB* boxA, OBB* boxB, WorldEntity* nodeA, WorldEntity* nodeB);


  protected:
    OBB*                bvElement;                  //!< the obb element
    OBBTreeNode*        nodePrev;                   //!< ref to the previous (parent) tree node = NULL if first
    OBBTreeNode*        nodeLeft;                   //!< ref to the left tree node
    OBBTreeNode*        nodeRight;                  //!< ref to the right tree node


  private:
    int                 treeIndex;                  //!< Index number of the BV in the tree
    int                 nextID;                     //!< the id of the next child
    int                 depth;                      //!< the depth of the node in the tree
    const OBBTree*      obbTree;                    //!< reference to the obb tree

    const modelInfo*    modelInf;                   //!< pointer to the models modelInfo object
    const int*          triangleIndexes;            //!< indexes to the used model triangles

    Plane               separationPlane;            //!< the separation plane of the obb
    sVec3D              sepPlaneCenter;             //!< only needed to draw plane
    int                 longestAxisIndex;           //!< only needed to draw plane

    /* tmp saving place for obb variables */
    int*                triangleIndexList1;         //!< pointer to the vert data of obbox1
    int*                triangleIndexList2;         //!< pointer to the vert data of obbox1
    int                 triangleIndexLength1;       //!< len vert data obbox1
    int                 triangleIndexLength2;       //!< len vert data obbox2

    WorldEntity*        owner;

};

#endif /* _OBB_TREE_NODE_H */
