/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "cruizer.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "util/loading/factory.h"
#include "key_mapper.h"
#include "state.h"

#include "graphics_engine.h"
#include "dot_emitter.h"
#include "sprite_particles.h"

#include "debug.h"

CREATE_FACTORY(Cruizer, CL_CRUIZER);

/**
 *  destructs the cruizer, deletes alocated memory
 */
Cruizer::~Cruizer ()
{
  this->setPlayer(NULL);
}

/**
 * @brief loads a Cruizer information from a specified file.
 * @param fileName the name of the File to load the cruizer from (absolute path)
 */
Cruizer::Cruizer(const std::string& fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for Cruizer.\n", fileName.c_str());
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 * @brief creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
Cruizer::Cruizer(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);


  this->getWeaponManager().changeWeaponConfig(1);
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

 // this->loadModel("models/ships/human_cruizer.obj", 0.05);

}


/**
 * @brief initializes a Cruizer
 */
void Cruizer::init()
{
  //  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_CRUIZER, "Cruizer");

  this->setSupportedPlaymodes(Playable::Full3D);

  bForward = bBackward = bLeft = bRight = bAscend = bDescend = false;
  mouseSensitivity = 0.005;

  this->rotorSpeed = 1000.0f;
  this->rotorCycle = 0.0f;
  this->cameraLook = 0.0f;
  this->rotation = 0.0f;
  this->acceleration = 10.0f;
  this->airFriction = 2.0f;

  this->setHealthMax(100);
  this->setHealth(100);


  // camera - issue
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  //this->cameraNode.setParentMode(PNODE_ROTATE_MOVEMENT);
  //this->cameraNode.setParent(this);


  // PARTICLES
  this->burstEmitter = new DotEmitter(200, 5.0, .01);
  this->burstEmitter->setParent(this);
  this->burstEmitter->setRelCoor(0, -0.7, 0);
  this->burstEmitter->setRelDir(Quaternion(-M_PI_2, Vector(0,0,1)));
  this->burstEmitter->setName("Cruizer_Burst_emitter_Left");


  this->burstSystem = new SpriteParticles(1000);
  this->burstSystem->addEmitter(this->burstEmitter);
  this->burstSystem->setName("SpaceShip_Burst_System");
  ((SpriteParticles*)this->burstSystem)->setMaterialTexture("maps/radial-trans-noise.png");
  this->burstSystem->setLifeSpan(1.0, .3);
  this->burstSystem->setRadius(0.0, 1.5);
  this->burstSystem->setRadius(0.05, 1.8);
  this->burstSystem->setRadius(.5, .8);
  this->burstSystem->setRadius(1.0, 0);
  this->burstSystem->setColor(0.0, .7,.7,1,.5);
  this->burstSystem->setColor(0.2, 0,0,0.8,.5);
  this->burstSystem->setColor(0.5, .5,.5,.8,.3);
  this->burstSystem->setColor(1.0, .8,.8,.8,.0);


  //add events to the eventlist of the Playable
  this->registerEvent(KeyMapper::PEV_FORWARD);
  this->registerEvent(KeyMapper::PEV_BACKWARD);
  this->registerEvent(KeyMapper::PEV_LEFT);
  this->registerEvent(KeyMapper::PEV_RIGHT);
  this->registerEvent(KeyMapper::PEV_UP);
  this->registerEvent(KeyMapper::PEV_DOWN);
  this->registerEvent(KeyMapper::PEV_FIRE1);
  this->registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  this->registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  this->registerEvent(EV_MOUSE_MOTION);

  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

  // WEAPON_MANAGER configuration
  this->getWeaponManager().setSlotCount(5);

  this->getWeaponManager().setSlotPosition(2, Vector(-1.63, .809, -.003));
  this->getWeaponManager().setSlotCapability(2, WTYPE_HEAVY);

  /// TODO: THESE ARE TOO MUCH
  this->getWeaponManager().setSlotPosition(3, Vector(-1.63, .678, -.652));
  this->getWeaponManager().setSlotDirection(3, Quaternion(-24/180 * M_PI, Vector(1,0,0)));

  this->getWeaponManager().setSlotPosition(4, Vector(-1.63, .678, .652));
  this->getWeaponManager().setSlotDirection(4, Quaternion(24/180 * M_PI, Vector(1,0,0)));

  this->cameraNode.setRelCoor(1,5,0);
  this->getWeaponManager().getFixedTarget()->setParent(&this->cameraNode);
  this->getWeaponManager().getFixedTarget()->setRelCoor(1000,0,0);

  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bAscend, &bAscend, "bAscend", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bDescend, &bDescend, "bDescend", PERMISSION_OWNER ) );
  //registerVar( new SynchronizeableQuaternion( &direction, &direction, "direction", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &cameraLook, &cameraLook, "cameraLook", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &rotation, &rotation, "rotation", PERMISSION_OWNER ) );
}

/**
 * @brief loads the Settings of a Cruizer from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void Cruizer::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);
}

void Cruizer::setPlayDirection(const Quaternion& rot, float speed)
{
  this->direction = Quaternion (rot.getHeading(), Vector(0,1,0));
}

void Cruizer::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( true);

  if (State::getCameraNode != NULL)
  {
    State::getCameraNode()->setParentSoft(&this->cameraNode);
    State::getCameraNode()->setRelCoorSoft(-10, 0,0);
    State::getCameraTargetNode()->setParentSoft(&this->cameraNode);
  }
}

void Cruizer::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
  this->detachCamera();

}


/**
 * @brief effect that occurs after the Cruizer is spawned
*/
void Cruizer::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 * @brief the action occuring if the cruizer left the game
*/
void Cruizer::leftWorld ()
{}

/**
 * @brief this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void Cruizer::collidesWith(WorldEntity* entity, const Vector& location)
{
  Playable::collidesWith(entity, location);
}



/**
 * @brief the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void Cruizer::tick (float dt)
{
  //  this->debugNode(1);
  Playable::tick(dt);

  // spaceship controlled movement
  this->movement(dt);
  this->rotorCycle += this->rotorSpeed * dt;

  // TRYING TO FIX PNode.
  this->cameraNode.setAbsCoorSoft(this->getAbsCoor() + Vector(0.0f, 5.0f, 0.0f), 30.0f);
  this->cameraNode.setRelDirSoft(this->getAbsDir(), 30.0f);
}

/**
 * @brief calculate the velocity
 * @param time the timeslice since the last frame
*/
void Cruizer::movement (float dt)
{
  Vector accel(0.0, 0.0, 0.0);

  if( this->bForward )
  {
    accel += Vector(this->acceleration, 0, 0);
  }

  if( this->bBackward )
  {
    accel -= Vector(this->acceleration, 0, 0);
  }
  if( this->bLeft)
  {
    accel -= Vector(0, 0, this->acceleration);
  }

  if( this->bRight)
  {
    accel += Vector(0, 0, this->acceleration);
  }

  if (this->bAscend )
  {
    accel += Vector(0, this->acceleration, 0);
  }
  if (this->bDescend )
  {
    accel -= Vector(0, this->acceleration, 0);
  }

  switch(this->getPlaymode())
  {
    case Playable::Full3D:
      {
        Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);

        // this is the air friction (necessary for a smooth control)
        Vector damping = (this->velocity * this->airFriction);


        this->velocity += (accelerationDir - damping)* dt;
        this->shiftCoor (this->velocity * dt);

        // limit the maximum rotation speed.
        if (this->rotation != 0.0f)
        {
          float maxRot = 10.0 * dt;
          if (unlikely(this->rotation > maxRot)) this->rotation = maxRot;
          if (unlikely(this->rotation < -maxRot)) this->rotation = -maxRot;
          this->direction *= Quaternion(-M_PI/4.0*this->rotation, Vector(0,1,0));

          this->rotation = 0.0f;
        }

        this->setRelDirSoft(this->direction * Quaternion(-cameraLook, Vector(0,0,1)), 5);
     }
      break;

    default:
      PRINTF(2)("Playmode %s Not Implemented\n", Playable::playmodeToString(this->getPlaymode()).c_str());
  }
}


void Cruizer::draw() const
{
  Vector tmpRot;
  WorldEntity::draw();
}

/**
 * @todo switch statement ??
 */
void Cruizer::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
    this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_UP)
    this->bAscend = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_DOWN)
    this->bDescend = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == KeyMapper::PEV_FORWARD)
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_BACKWARD)
    this->bBackward = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    float xMouse, yMouse;
    xMouse = event.xRel*mouseSensitivity;
    yMouse = event.yRel*mouseSensitivity;

    // rotate the Player around the y-axis
    this->rotation += xMouse;

    this->cameraLook += yMouse;
    // rotate the Camera around the z-axis
    if (cameraLook > M_PI_4)
      cameraLook = M_PI_4;
    else if (cameraLook < -M_PI_4)
      cameraLook = -M_PI_4;
    //this->cameraNode.setRelDirSoft(this->direction,10);
  }
}
