/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: ...

*/

#include "fps_player.h"

#include "interactive_model.h"
#include "state.h"

#include "src/lib/util/loading/factory.h"

#include "md2/md2Model.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"
#include "weapons/fps_sniper_rifle.h"
#include "weapons/aiming_system.h"

#include "aabb.h"

#include "key_mapper.h"

#include "debug.h"

#include "shared_network_data.h"



CREATE_FACTORY(FPSPlayer, CL_FPS_PLAYER);

#include "script_class.h"
CREATE_SCRIPTABLE_CLASS(FPSPlayer, CL_FPS_PLAYER,
                        addMethod("setAbsCoor", ExecutorLua3<PNode,float,float,float>(&PNode::setAbsCoor))
                            ->addMethod("getAbsCoorX", ExecutorLua0ret<PNode, float>(&PNode::getAbsCoorX))
                            ->addMethod("getAbsCoorY", ExecutorLua0ret<PNode, float>(&PNode::getAbsCoorY))
                            ->addMethod("getAbsCoorZ", ExecutorLua0ret<PNode, float>(&PNode::getAbsCoorZ))
                       );


/**
 *  destructs the FPSPlayer, deletes alocated memory
 */
FPSPlayer::~FPSPlayer ()
{
  this->setPlayer(NULL);
}


/**
 *  creates a new FPSPlayer from Xml Data
 * @param root the xml element containing FPSPlayer data
 *
 */
FPSPlayer::FPSPlayer(const TiXmlElement* root)
{
  this->init();

  if (root != NULL)
    this->loadParams(root);

}


/**
 * initializes a FPSPlayer
 */
void FPSPlayer::init()
{
  this->setClassID(CL_FPS_PLAYER, "FPSPlayer");


  this->bLeft = false;
  this->bRight = false;
  this->bForward = false;
  this->bBackward = false;
  this->bJump = false;
  this->bPosBut = false;

  this->xMouse = 0.0f;
  this->yMouse = 0.0f;

  this->setHealthMax(100);
  this->setHealth(80);

  this->fallVelocity = 0.0f;
  this->jumpForce = 0.0f;

  this->cameraNode.setParent(this);

  this->attitude = this->getAbsDir().getAttitude();
  this->heading = this->getAbsDir().getHeading();

  //add events to the eventlist
  registerEvent(KeyMapper::PEV_FORWARD);
  registerEvent(KeyMapper::PEV_BACKWARD);
  registerEvent(KeyMapper::PEV_LEFT);
  registerEvent(KeyMapper::PEV_RIGHT);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_JUMP);
  registerEvent(EV_MOUSE_MOTION);



  // weapon manager for the fps
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

  Weapon* wpRight = new FPSSniperRifle(0);
  wpRight->setName("testGun Right");
/*  Weapon* wpLeft = new TestGun(1);*/
//   Weapon* wpLeft = new Turret();
//   wpLeft->setName("testGun Left");

//   this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1, 0);
  this->getWeaponManager().changeWeaponConfig(1);

  this->getWeaponManager().setSlotCount(2);
//   this->getWeaponManager().setSlotDirection(0, Quaternion(M_PI_2, Vector(0,1,0)));
  this->getWeaponManager().setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->getWeaponManager().setSlotDirection(1, Quaternion(M_PI_4*.5, Vector(1,0,0)));
  this->getWeaponManager().setSlotPosition(0, Vector(1.5, -0.7, 1.1));
  this->getWeaponManager().setSlotPosition(1, Vector(5.0, 0.0, 0.0));


  this->getWeaponManager().setParentNode(&this->cameraNode);
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);

  this->getWeaponManager().getFixedTarget()->setParent(&this->cameraNode);
  this->getWeaponManager().getFixedTarget()->setRelCoor(1000,0,0);


  // network registration
  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bJump, &bJump, "bJump", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &heading, &heading, "heading", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &attitude, &attitude, "attitude", PERMISSION_OWNER ) );

    //subscribe to collision reaction
  this->subscribeReaction(CREngine::CR_PHYSICS_FULL_WALK, CL_BSP_ENTITY);

  this->initWeapon = false;
}


/**
 * loads the Settings of a FPSPlayer from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void FPSPlayer::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);
}

void FPSPlayer::setPlayDirection(const Quaternion& quat, float speed)
{
  this->attitude = this->getAbsDir().getAttitude();
  this->heading = this->getAbsDir().getHeading();
}


void FPSPlayer::reset()
{
  this->bLeft = false;
  this->bRight = false;
  this->bForward = false;
  this->bBackward = false;
  this->xMouse = 0.0f;
  this->yMouse = 0.0f;

  this->setHealth(80);
}


void FPSPlayer::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( true );

  State::getCameraNode()->setParentSoft(&this->cameraNode);
  State::getCameraTargetNode()->setParentSoft(&this->cameraNode);

  this->getWeaponManager().getFixedTarget()->setParent(State::getCameraTargetNode());
  this->getWeaponManager().getFixedTarget()->setRelCoor(0,0,0);

  if ( !State::isOnline() )
  {
    this->respawn();
  }
}

void FPSPlayer::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
  this->detachCamera();
}



/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
 */
void FPSPlayer::tick (float time)
{

  if ( !this->initWeapon )
  {
    this->initWeapon = true;

    AABB* box = this->getModelAABB();
    if( box != NULL)
    {
      float f = 1.3f;
      State::getCameraNode()->setRelCoor(0, box->halfLength[1] * f, 0);
      State::getCameraTargetNode()->setRelCoor(10, box->halfLength[1] * f, 0);

      State::getCameraNode()->setParentMode(PNODE_ROTATE_AND_MOVE);


      this->getWeaponManager().setSlotPosition(0, Vector(1.5, box->halfLength[1] * f - 0.7, 1.1));
      this->getWeaponManager().setSlotPosition(1, Vector(5.0, box->halfLength[1] * f, 0.0));
    }
  }

  if( this->bJump)
  {
    printf("mechanic2:walkTo( %f, mtheight, %f)\n",this->getAbsCoorX(),this->getAbsCoorZ());
  }




  Playable::tick( time );

  if( ( xMouse != 0 || yMouse != 0 ) && (this->getOwner() == SharedNetworkData::getInstance()->getHostID() || !State::isOnline() ) )
  {
    xMouse *= time ;
    yMouse *= time ;

    heading -= xMouse;
    attitude-= yMouse;


    if ( attitude > 0.32 )
      attitude = 0.32;

    else if ( attitude < -2.1 )
      attitude = -2.1;

    xMouse = yMouse = 0;
  }

  this->setAbsDir(Quaternion(heading, Vector(0,1,0)));
  this->cameraNode.setRelDir(Quaternion( attitude, Vector( 0, 0, 1 ) ));

  Vector velocity;

  if ( this->bForward )
  {
    velocity += this->getAbsDirX();
  }

  if ( this->bBackward )
  {
    velocity -= this->getAbsDirX();
  }

  if ( this->bRight )
  {
    velocity += this->getAbsDirZ();
  }

  if ( this->bLeft )
  {
    velocity -= this->getAbsDirZ();
  }


  velocity *= 100;

  if( this->bJump && likely(this->getModel(0) != NULL))
  {
    if( this->jumpForce < 1.0f)
    {
      this->jumpForce = 300.0f;

      if( ((InteractiveModel*)this->getModel(0))->getAnimation() != JUMP)
        ((InteractiveModel*)this->getModel(0))->setAnimation(JUMP);
    }
  }
  else if(velocity.len() != 0.0f)
  {
    if( ((InteractiveModel*)this->getModel(0))->getAnimation() != RUN)
      ((InteractiveModel*)this->getModel(0))->setAnimation(RUN);
  }
  else
  {
    if( ((InteractiveModel*)this->getModel(0))->getAnimation() != STAND)
      ((InteractiveModel*)this->getModel(0))->setAnimation(STAND);
  }


  velocity.y += this->jumpForce;
  if( this->jumpForce > 1.0f)
    this->jumpForce *= 0.9f;


  // physical falling of the player
  if( !this->isOnGround())
  {
    this->fallVelocity += 300.0f * time;
    velocity -= Vector(0.0, 1.0, 0.0) * this->fallVelocity;
  }
  else
  {
    this->fallVelocity = 0.0f;
  }

  this->shiftCoor( velocity*time );







  if( likely(this->getModel(0) != NULL) && this->getModel(0)->isA(CL_INTERACTIVE_MODEL))
  {
    ((InteractiveModel*)this->getModel(0))->tick(time);

    // handle animations differently





//     else if( this->bFire && likely(this->getModel(0) != NULL))
//     {
//       if( ((InteractiveModel*)this->getModel(0))->getAnim() != ATTACK)
//         ((InteractiveModel*)this->getModel(0))->setAnimation(ATTACK);
//     }
//     else if( fabs(move.len()) > 0.0f && likely(this->getModel(0) != NULL))
//     {
//       if( ((InteractiveModel*)this->getModel(0))->getAnim() != RUN)
//         ((InteractiveModel*)this->getModel(0))->setAnimation(RUN);
//     }
//     else if (likely(this->getModel(0) != NULL))
//     {
//       if( ((InteractiveModel*)this->getModel(0))->getAnimation() != STAND)
//         ((InteractiveModel*)this->getModel(0))->setAnimation(STAND);
//     }
  }

  this->setOnGround(false);

}



/**
 *  draws the MD2Creature after transforming it.
 */
void FPSPlayer::draw () const
{
  // only draw if this entity is not the player since the player nevers sees himself
  if( this->getCurrentPlayer() == NULL)
    WorldEntity::draw();
}



/**
 * process
 */
void FPSPlayer::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
    this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_FORWARD)
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_BACKWARD)
    this->bBackward = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse += event.xRel;
    this->yMouse += event.yRel;
  }
  else if( event.type == KeyMapper::PEV_JUMP)
    this->bJump = event.bPressed;
}


void FPSPlayer::respawn( )
{
  if( State::isOnline())
    toList( OM_PLAYERS );
}

void FPSPlayer::destroy( WorldEntity* killer )
{
  Playable::destroy( killer );
  
  toList( OM_DEAD );
}

