/*
	orxonox - the future of 3D-vertical-scrollers

	Copyright (C) 2004 orx

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

### File Specific:
	main-programmer: hdavid, amaechler
*/

#include "snow_effect.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "util/loading/resource_manager.h"

#include "glincl.h"
#include "debug.h"

#include "p_node.h"
#include "state.h"
#include "sprite_particles.h"
#include "plane_emitter.h"
#include "shell_command.h"
#include "script_class.h"

#include "parser/tinyxml/tinyxml.h"

SHELL_COMMAND(activate, SnowEffect, activateSnow);
SHELL_COMMAND(deactivate, SnowEffect, deactivateSnow);

using namespace std;

CREATE_SCRIPTABLE_CLASS(SnowEffect, CL_SNOW_EFFECT,
                            addMethod("activate", ExecutorLua0<SnowEffect>(&SnowEffect::activate))
                            ->addMethod("deactivate", ExecutorLua0<SnowEffect>(&SnowEffect::deactivate))
                       );

CREATE_FACTORY(SnowEffect, CL_SNOW_EFFECT);

SnowEffect::SnowEffect(const TiXmlElement* root)
{
	this->setClassID(CL_SNOW_EFFECT, "SnowEffect");

	this->init();

	if (root != NULL)
		this->loadParams(root);

	//load wind sound
	if (this->snowWindForce > 1) {
		if (this->windBuffer != NULL)
			ResourceManager::getInstance()->unload(this->windBuffer);
	       		this->windBuffer = (OrxSound::SoundBuffer*)ResourceManager::getInstance()->load("sound/atmosphere/wind.wav", WAV);
	}

  if(snowActivate) {
		this->activate();
    SnowEffect::snowParticles->precache((int) this->snowLife);
  }
}


SnowEffect::~SnowEffect()
{
	this->deactivate();
}

SpriteParticles* SnowEffect::snowParticles = NULL;

void SnowEffect::loadParams(const TiXmlElement* root)
{
	WeatherEffect::loadParams(root);

	LoadParam(root, "numParticles", this, SnowEffect, numParticles);
	LoadParam(root, "materialTexture", this, SnowEffect, materialTexture);
	LoadParam(root, "lifeSpans", this, SnowEffect, lifeSpan);
	LoadParam(root, "radius", this, SnowEffect, radius);
	LoadParam(root, "mass", this, SnowEffect, mass);
	LoadParam(root, "emissionRate", this, SnowEffect, emissionRate);
	LoadParam(root, "emissionVelocity", this, SnowEffect, emissionVelocity);
	LoadParam(root, "wind", this, SnowEffect, wind);
	LoadParam(root, "size", this, SnowEffect, size);
	LoadParam(root, "coord", this, SnowEffect, coord);

	LOAD_PARAM_START_CYCLE(root, element);
	{
		LoadParam_CYCLE(element, "option", this, SnowEffect, setSnowOption);
	}
	LOAD_PARAM_END_CYCLE(element);
}

void SnowEffect::init()
{
	this->emitter = new PlaneEmitter();

	// Default values
	this->snowActivate = false;
	this->snowMove = false;
	this->particles = 12000;
	this->texture = "maps/snow_flake_01_32x32.png";
	this->snowLife = 8;
	this->randomLife = 2;
	this->snowRadius = 3.5;
	this->randomRadius = 1;
	this->snowMass = 1.0;
	this->randomMass = 0.3;
	this->rate = 900;
	this->velocity = -100;
	this->randomVelocity = 5;
	this->angle = 0.5;
	this->randomAngle = 0.2;
	this->alpha = 0.5;
	this->snowSize = Vector2D(2500, 2500);
	this->snowCoord = Vector(100,450,400);
	this->snowWindForce = 1;
}

void SnowEffect::activate()
{
	PRINTF(3)("Activating SnowEffect\n");

	this->snowActivate = true;

	SnowEffect::snowParticles = new SpriteParticles(particles);
	SnowEffect::snowParticles->setName("SnowEffectTrailParticles");
	SnowEffect::snowParticles->setMaterialTexture(texture);
	SnowEffect::snowParticles->setLifeSpan(snowLife, randomLife);
	SnowEffect::snowParticles->setRadius(0.0, snowRadius, randomRadius);
	SnowEffect::snowParticles->setRadius(0.2, snowRadius, randomRadius*0.8);
	SnowEffect::snowParticles->setRadius(1.0, snowRadius, randomRadius*0.5);
	SnowEffect::snowParticles->setMass(0, snowMass, randomMass);
	SnowEffect::snowParticles->setColor(0,1, 1, 1, alpha);
	SnowEffect::snowParticles->setColor(.5, .6, .6, .6, alpha/2);
	SnowEffect::snowParticles->setColor(1, .0, .0, .0, .0);

	this->emitter->setSystem(SnowEffect::snowParticles);

	this->emitter->setRelCoor(snowCoord);
	this->emitter->setEmissionRate(rate);
	this->emitter->setEmissionVelocity(velocity, randomVelocity);
	this->emitter->setSpread(angle * this->snowWindForce , randomAngle * this->snowWindForce);
	this->emitter->setSize(snowSize);

  if (this->snowWindForce != 0)
    this->soundSource.play(this->windBuffer, 0.1f * this->snowWindForce, true);

}


void SnowEffect::deactivate()
{
	PRINTF(3)("Deactivating SnowEffect\n");

	this->snowActivate = false;
	this->emitter->setSystem(NULL);

	if (this->windBuffer != NULL)
		ResourceManager::getInstance()->unload(this->windBuffer);
}

void SnowEffect::draw() const
{
	if (!this->snowActivate)
		return;
}

void SnowEffect::tick(float dt)
{
	if (!this->snowActivate)
		return;

	/*
	float distance = (State::getCameraNode()->getAbsCoor() - Vector(snowCoord.x, State::getCameraNode()->getAbsCoor().y, snowCoord.z)).len();

	if(activated)
	{
	if(distance > 0.3*snowSize.x || distance > 0.3*snowSize.y)
			this->deactivate();
	else if(distance > 0.25*snowSize.x || distance > 0.25*snowSize.y)
	this->alpha = 0.15;
	else if(distance > 0.2*snowSize.x || distance > 0.2*snowSize.y)
	this->alpha = 0.25;
	else if(distance > 0.1*snowSize.x || distance > 0.1*snowSize.y)
	this->alpha = 0.4;

	SnowEffect::snowParticles->setColor(0,1, 1, 1, alpha);
	SnowEffect::snowParticles->setColor(.5, .6, .6, .6, alpha/2);
	SnowEffect::snowParticles->setColor(1, .0, .0, .0, .0);
	}
	else
	{
	if(distance < 0.3*snowSize.x || distance < 0.3*snowSize.y )
	this->activate();
	if( distance < 0.25*snowSize.x || distance < 0.25*snowSize.y )
	this->alpha = 0.25;
	else if( distance < 0.2*snowSize.x || distance < 0.2*snowSize.y )
	this->alpha = 0.4;
	else if( distance < 0.1*snowSize.x || distance < 0.1*snowSize.y )
	this->alpha = 0.5;

	SnowEffect::snowParticles->setColor(0,1, 1, 1, alpha);
	SnowEffect::snowParticles->setColor(.5, .6, .6, .6, alpha/2);
	SnowEffect::snowParticles->setColor(1, .0, .0, .0, .0);
	}*/

	if (this->snowMove) {
		this->snowCoord = State::getCameraNode()->getAbsCoor();
		this->emitter->setRelCoor(this->snowCoord.x , this->snowCoord.y+300, this->snowCoord.z);
	}
}
