/*
  orxonox - the future of 3D-vertical-scrollers

  Copyright (C) 2004 orx

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

### File Specific:
  main-programmer: hdavid, amaechler
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "rain_effect.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "util/loading/resource_manager.h"

#include "glincl.h"
#include "p_node.h"
#include "state.h"
#include "spark_particles.h"
#include "plane_emitter.h"
#include "shell_command.h"
#include "light.h"
#include "cloud_effect.h"
#include "script_class.h"

#include "parser/tinyxml/tinyxml.h"

// Define shell commands
SHELL_COMMAND(activate, RainEffect, activateRain);
SHELL_COMMAND(deactivate, RainEffect, deactivateRain);
SHELL_COMMAND(startraining, RainEffect, startRaining);
SHELL_COMMAND(stopraining, RainEffect, stopRaining);

using namespace std;

CREATE_SCRIPTABLE_CLASS(RainEffect, CL_RAIN_EFFECT,
                        addMethod("startRaining", ExecutorLua0<RainEffect>(&RainEffect::startRaining))
                            ->addMethod("stopRaining", ExecutorLua0<RainEffect>(&RainEffect::stopRaining))
                            ->addMethod("activate", ExecutorLua0<RainEffect>(&RainEffect::activate))
                            ->addMethod("deactivate", ExecutorLua0<RainEffect>(&RainEffect::deactivate))
                       );

CREATE_FACTORY(RainEffect, CL_RAIN_EFFECT);

/**
 * @brief standard constructor
 */
RainEffect::RainEffect(const TiXmlElement* root) {
    this->setClassID(CL_RAIN_EFFECT, "RainEffect");

    this->init();

    if (root != NULL)
        this->loadParams(root);

    //load rain sound
    if (this->rainBuffer != NULL)
        ResourceManager::getInstance()->unload(this->rainBuffer);
    this->rainBuffer = (OrxSound::SoundBuffer*)ResourceManager::getInstance()->load("sound/atmosphere/rain.wav", WAV);

    //load wind sound
    if (this->rainWindForce != 0) {
        if (this->windBuffer != NULL)
            ResourceManager::getInstance()->unload(this->windBuffer);
        this->windBuffer = (OrxSound::SoundBuffer*)ResourceManager::getInstance()->load("sound/atmosphere/wind.wav", WAV);
    }

    if(rainActivate) {
        this->activate();
        RainEffect::rainParticles->precache((int)this->rainLife * 2);
    }
}

/**
 * @brief standard deconstructor
 */
RainEffect::~RainEffect() {
    this->deactivate();

    if (this->rainBuffer != NULL)
        ResourceManager::getInstance()->unload(this->rainBuffer);

    if (this->windBuffer != NULL)
        ResourceManager::getInstance()->unload(this->windBuffer);
}

/**
 * @brief initalizes the rain effect with default values
 */
void RainEffect::init() {
    //Default values
    this->rainActivate = false;
    this->rainFadeInActivate = false;
    this->rainFadeOutActivate = false;

    this->rainMove = false;
    this->rainCoord = Vector(500, 500, 500);
    this->rainSize = Vector2D(1000, 1000);
    this->rainRate = 4000;
    this->rainVelocity = -300;
    this->rainLife = 4;
    this->rainWindForce  = 0;
    this->rainFadeInDuration = 10;
    this->rainFadeOutDuration = 10;

    this->cloudColor = Vector(0.6f, 0.6f, 0.6f);
    this->skyColor = Vector(0.0f, 0.0f, 0.0f);

    this->rainMaxParticles = this->rainRate * this->rainLife;
    this->localTimer = 0;
    this->soundRainVolume = 0.3f;
    this->emitter = new PlaneEmitter(this->rainSize);

}

/**
 * @brief loads the rain effect parameters.
 * @param root: the XML-Element to load the data from
 */
void RainEffect::loadParams(const TiXmlElement* root) {
    WeatherEffect::loadParams(root);

    LoadParam(root, "coord", this, RainEffect, setRainCoord);
    LoadParam(root, "size", this, RainEffect, setRainSize);
    LoadParam(root, "rate", this, RainEffect, setRainRate);
    LoadParam(root, "velocity", this, RainEffect, setRainVelocity);
    LoadParam(root, "life", this, RainEffect, setRainLife);
    LoadParam(root, "wind", this, RainEffect, setRainWind);
    LoadParam(root, "fadeinduration", this, RainEffect, setRainFadeIn);
    LoadParam(root, "fadeoutduration", this, RainEffect, setRainFadeOut);
    LoadParam(root, "cloudcolor", this, RainEffect, setCloudColor);
    LoadParam(root, "skycolor", this, RainEffect, setSkyColor);

    LOAD_PARAM_START_CYCLE(root, element);
    {
        LoadParam_CYCLE(element, "option", this, RainEffect, setRainOption);
    }
    LOAD_PARAM_END_CYCLE(element);
}

SparkParticles* RainEffect::rainParticles = NULL;

/**
 * @brief activates the rain effect
 */
void RainEffect::activate() {
    PRINTF(3)( "Activating RainEffect, coord: %f, %f, %f, size: %f, %f, rate: %f, velocity: %f, moveRain: %s\n", this->rainCoord.x, this->rainCoord.y, this->rainCoord.z, this->rainSize.x, this-> rainSize.y, this->rainRate, this->rainVelocity, this->rainMove ? "true" : "false" );

    this->rainActivate = true;

    if (unlikely(RainEffect::rainParticles == NULL)) {
        RainEffect::rainParticles = new SparkParticles((int) this->rainMaxParticles);
        RainEffect::rainParticles->setName("RainParticles");
        RainEffect::rainParticles->setLifeSpan(this->rainLife, 2);
        RainEffect::rainParticles->setRadius(0, 0.03);
        RainEffect::rainParticles->setRadius(0.2, 0.02);
        RainEffect::rainParticles->setRadius(1, 0.01);
        RainEffect::rainParticles->setColor(0, 0.3, 0.3, 0.5, 0.2);   // grey blue 1
        RainEffect::rainParticles->setColor(0.5, 0.4, 0.4, 0.5, 0.2); // grey blue 2
        RainEffect::rainParticles->setColor(1, 0.7, 0.7, 0.7, 0.2);   // light grey
    }

    this->emitter->setSystem(RainEffect::rainParticles);
    this->emitter->setRelCoor(this->rainCoord);
    this->emitter->setEmissionRate(this->rainRate);
    this->emitter->setEmissionVelocity(this->rainVelocity);
    this->emitter->setSpread(this->rainWindForce / 50, 0.2);

    // play rain sound and loop it
    this->soundSource.play(this->rainBuffer, this->soundRainVolume, true);

    // if there's wind, play wind sound
    if (this->rainWindForce > 0)
        this->soundSource.play(this->windBuffer, 0.1f * this->rainWindForce, true);

    // Store cloud- and sky color before the rain;
    this->oldCloudColor = CloudEffect::cloudColor;
    this->oldSkyColor   = CloudEffect::skyColor;

    // If we're not fading, change color immediately
    if (!this->rainFadeInActivate || !this->rainFadeOutActivate) {
        CloudEffect::changeCloudColor(this->cloudColor, 0);
        CloudEffect::changeSkyColor(this->skyColor, 0);
    }
}

/**
 * @brief deactivates the rain effect
 */
void RainEffect::deactivate() {
    PRINTF(3)("Deactivating RainEffect\n");

    this->rainActivate = false;
    this->rainFadeInActivate = false;
    this->rainFadeOutActivate = false;

    this->emitter->setSystem(NULL);

    // Stop Sound
    this->soundSource.stop();

    // Restore the old cloud- and sky color
    CloudEffect::changeCloudColor(this->oldCloudColor, 0);
    CloudEffect::changeSkyColor(this->oldSkyColor, 0);
}

/**
 * @brief ticks the rain effect
 * @param dt: tick float
 */
void RainEffect::tick (float dt) {
    if (!this->rainActivate)
        return;

    if (this->rainMove) {
        this->rainCoord = State::getCameraNode()->getAbsCoor();
        this->emitter->setRelCoor(this->rainCoord.x , this->rainCoord.y+800, this->rainCoord.z);
    }

    if (this->rainFadeInActivate) {
        PRINTF(4)("tick fading IN RainEffect\n");

        this->localTimer += dt;
        float progress = this->localTimer / this->rainFadeInDuration;

        // use alpha in color to fade in rain
        RainEffect::rainParticles->setColor(0,   0.3, 0.3, 0.5, 0.2 * progress); // grey blue 1
        RainEffect::rainParticles->setColor(0.5, 0.4, 0.4, 0.5, 0.2 * progress); // grey blue 2
        RainEffect::rainParticles->setColor(1,   0.7, 0.7, 0.7, 0.2 * progress); // light grey

        // increase radius for more "heavy" rain
        RainEffect::rainParticles->setRadius(0, 0.03 * progress);
        RainEffect::rainParticles->setRadius(0.2, 0.02 * progress);
        RainEffect::rainParticles->setRadius(1, 0.01 * progress);

        // increase sound volume
        if (!this->soundSource.isPlaying())
            this->soundSource.play(this->rainBuffer, this->soundRainVolume, true);
        this->soundSource.gain(this->rainBuffer, this->soundRainVolume * progress);

        if (progress >= 1)
            this->rainFadeInActivate = false;
    }

    if (this->rainFadeOutActivate) {
        PRINTF(4)("tick fading OUT RainEffect\n");

        this->localTimer += dt;
        float progress = 1 - (this->localTimer / this->rainFadeOutDuration);

        // use alpha in color to fade out
        RainEffect::rainParticles->setColor(0,   0.3, 0.3, 0.5, 0.2 * progress); // grey blue 1
        RainEffect::rainParticles->setColor(0.5, 0.4, 0.4, 0.5, 0.2 * progress); // grey blue 2
        RainEffect::rainParticles->setColor(1,   0.7, 0.7, 0.7, 0.2 * progress); // light grey

        // decrease radius
        RainEffect::rainParticles->setRadius(0, 0.03 * progress);
        RainEffect::rainParticles->setRadius(0.2, 0.02 * progress);
        RainEffect::rainParticles->setRadius(1, 0.01 * progress);

        // decrease sound volume
        if (!this->soundSource.isPlaying())
            this->soundSource.play(this->rainBuffer, this->soundRainVolume, true);
        this->soundSource.gain(this->rainBuffer, this->soundRainVolume * progress);

        if (progress <= 0) {
            PRINTF(4)("tick fading OUT RainEffect COMPLETED! Deactivating...\n");
            this->rainFadeOutActivate = false;
            this->deactivate();
        }
    }
}

/**
 * @brief starts raining slowly
*/
void RainEffect::startRaining() {
    PRINTF(4)("startRaining function;\n");

    // If it is already raining, do nothing
    if (this->rainActivate)
        return;

    this->localTimer = 0;
    this->rainFadeInActivate = true;

    PRINTF(4)("startRaining function complete; fadedur: %f\n", this->rainFadeInDuration);
    this->activate();

    CloudEffect::changeCloudColor(this->cloudColor, this->rainFadeInDuration);
    CloudEffect::changeSkyColor(this->skyColor, this->rainFadeInDuration);
}

/**
 * @brief stops raining slowly
 */
void RainEffect::stopRaining() {
    PRINTF(4)("stopRaining function;\n");

    // If it is not raining, do nothing
    if (!this->rainActivate)
        return;

    this->localTimer = 0;
    this->rainFadeOutActivate = true;

    PRINTF(4)("stopRaining function completed; fadedur: %f\n", this->rainFadeOutDuration);

    CloudEffect::changeCloudColor(this->oldCloudColor, this->rainFadeOutDuration);
    CloudEffect::changeSkyColor(this->oldSkyColor, this->rainFadeOutDuration);
}

/**
 * @brief hides the rain
 */
void RainEffect::hideRain() {
    RainEffect::rainParticles->setColor(0, 0,0,0, 0);
    RainEffect::rainParticles->setColor(0, 0,0,0, 0);
}
