 /*!
 * @file mapped_water.h
 *  worldentity for flat, cool looking, mapped water
*/
/*! example input in .oxw file with the standard values
<MappedWater>
  <waterpos>0,0,0</waterpos>
  <watersize>100,100</watersize>
  <wateruv>9</wateruv><!-- size of the waves -->
  <waterflow>0.08</waterflow>
  <lightpos>0,10,0</lightpos>
  <waterangle>0</waterangle>
  <normalmapscale>0.25</normalmapscale><!-- you won't see a big differnce if you change that -->
  <shinesize>128</shinesize><!-- the bigger the value, the smaller the specular reflection point -->
  <shinestrength>0.7</shinestrength>
  <reflstrength>1</reflstrength>
  <refraction>0.009</refraction>
  <watercolor>0.1, 0.2, 0.4</watercolor>
</MappedWater>
*/


#ifndef _MAPPED_WATER_H
#define _MAPPED_WATER_H

#include "world_entity.h"
#include "material.h"
#include "shader.h"

namespace OrxGui { class GLGuiBox; };

// forward declaration
template <class T> class tAnimation;

class MappedWater : public WorldEntity
{
public:
  MappedWater(const TiXmlElement* root = NULL);
  virtual ~MappedWater();

  // worldentity functions
  void loadParams(const TiXmlElement* root);
  void saveParams();
  void draw() const;
  void tick(float dt);

  // function to prepare renderpaths for creation of refleaction and reflaction textures
  void activateReflection();
  void deactivateReflection();
  void activateRefraction();
  void deactivateRefraction();

  // slider gui to edit water params during gameplay
  void toggleGui();

  // functions to set parameters for the water, usually they're called through loadparam
  void setLightPos(float x, float y, float z) { this->lightPos = Vector(x,y,z); }
  void setWaterPos(float x, float y, float z) { this->waterHeight = y; this->waterVerts[0] = x; this->waterVerts[1] = z; }
  void setWaterHeight(float y) { this->waterHeight = y;  }
  void setWaterSize(float x, float z) { this->xWidth = x; this->zWidth = z; }
  void setWaterAngle(float angle) { this->waterAngle = angle; }
  void setWaterColor(float r, float g, float b) { this->waterColor = Vector(r,g,b); }
  void setWaterUV(float uv) { this->waterUV = uv; }
  void setWaterFlow(float flow) { this->waterFlow = flow; }
  void setNormalMapScale(float scale) { this->kNormalMapScale = scale; }
  void setShineSize(float shine) { this->shineSize = shine; }
  void setShineStrength(float strength) { this->shineStrength = strength; }
  void setReflStrength(float strength) { this->reflStrength = strength; }
  void setRefraction(float refraction) { this->refraction = refraction; }

  // functions to change water parameters during runtime
  // to reset waterUV and waterFlow just use the normal set functions
  // don't reset kNormalMapScale (because it won't change anything)
  void resetLightPos(float x, float y, float z);
  void resetLightPosX(float x) { this->resetLightPos(x, this->lightPos.y, this->lightPos.z); }
  void resetLightPosY(float y) { this->resetLightPos(this->lightPos.x, y, this->lightPos.z); }
  void resetLightPosZ(float z) { this->resetLightPos(this->lightPos.x, this->lightPos.y, z); }
  void resetWaterPos(float x, float y, float z) { this->setWaterPos(x, y, z); this->calcVerts(); }
  void resetWaterSize(float x, float z) { this->setWaterSize(x, z); this->calcVerts(); }
  void resetWaterAngle(float angle) { this->setWaterAngle(angle); this->calcVerts(); }
  void resetWaterColor(float r, float g, float b);
  void resetWaterColorR(float r) { this->resetWaterColor(r, this->waterColor.y, this->waterColor.z); }
  void resetWaterColorG(float g) { this->resetWaterColor(this->waterColor.x, g, this->waterColor.z); }
  void resetWaterColorB(float b) { this->resetWaterColor(this->waterColor.x, this->waterColor.y, b); }
  void resetShineSize(float shine);
  void resetShineStrength(float strength);
  void resetReflStrength(float strength);
  void resetRefraction(float refraction);

  // fading functions
  void fadeWaterUV(float uv, float time) { this->newWaterUV = uv; this->waterUVFadeTime = time; this->bFadeWaterUV = true; }
  void fadeWaterFlow(float flow, float time) { this->newWaterFlow = flow; this->waterFlowFadeTime = time; this->bFadeWaterFlow = true; }
  void fadeShineSize(float shine, float time) { this->newShineSize = shine; this->shineSizeFadeTime = time; this->bFadeShineSize = true; }
  void fadeShineStrength(float strength, float time) { this->newShineStrength = strength; this->shineStrengthFadeTime = time; this->bFadeShineStrength = true; }
  void fadeReflStrength(float strength, float time) { this->newReflStrength = strength; this->reflStrengthFadeTime = time; this->bFadeReflStrength = true; }
  void fadeRefraction(float refraction, float time) { this->newRefraction = refraction; this->refractionFadeTime = time; this->bFadeRefraction = true; }
  void fadeWaterHeight(float y, float time) { this->newWaterHeight = y; this->waterHeightFadeTime = time; this->bFadeWaterHeight = true; }
  void fadeWaterColor(float r, float g, float b, float time) { this->newWaterColor = Vector(r, g, b); this->waterColorFadeTime = time; this->bFadeWaterColor = true; }

private:
  void initParams();
  void initTextures();
  void initShaders();
  void calcVerts();

private:
  Material                  mat;
  Shader*                   shader;
  OrxGui::GLGuiBox*         box;

  // water size and position
  float                     waterHeight;            //!< position of the water
  float                     waterVerts[8];          //!< coords of the 4 vertexes of the water quad
  float                     xWidth, zWidth;         //!< size of the water quad
  float                     waterAngle;             //!< defines how much the water will be turned around the point waterPos

  // values for texture size, scale, texture coords
  float                     move;
  float                     move2;
  float                     waterUV;                //!< size of the waves
  float                     waterFlow;              //!< speed of the water
  float                     normalUV;
  float                     kNormalMapScale;
  int                       textureSize;            //!< height and width of the texture

  // values for the uniforms
  Vector                    waterColor;             //!< color of the water
  Vector                    lightPos;               //!< position of the light that is used to render the reflection
  float                     shineSize;              //!< the bigger the value, the smaller the specular reflection point
  float                     shineStrength;
  float                     reflStrength;
  float                     refraction;

  // uniforms
  Shader::Uniform*          cam_uni;                //!< uniform that is used for the camera position
  Shader::Uniform*          light_uni;              //!< uniform that is used for the light position
  Shader::Uniform*          color_uni;              //!< uniform that is used for the watercolor
  Shader::Uniform*          shineSize_uni;          //!< uniform that is used for the specular shininessd of the water
  Shader::Uniform*          shineStrength_uni;      //!< uniform that is used for the strenght of the specular reflection
  Shader::Uniform*          reflStrength_uni;       //!< uniform that is used for the strength of the reflection
  Shader::Uniform*          refr_uni;               //!< uniform that is used for the strength of the refraction

  // fading TODO fix this so it isnt so hacky anymore
  tAnimation<MappedWater>*  waterUVFader;
  float                     newWaterUV;
  float                     waterUVFadeTime;
  bool                      bFadeWaterUV ;
  tAnimation<MappedWater>*  waterFlowFader;
  float                     newWaterFlow;
  float                     waterFlowFadeTime;
  bool                      bFadeWaterFlow;
  tAnimation<MappedWater>*  shineSizeFader;
  float                     newShineSize;
  float                     shineSizeFadeTime;
  bool                      bFadeShineSize;
  tAnimation<MappedWater>*  shineStrengthFader;
  float                     newShineStrength;
  float                     shineStrengthFadeTime;
  bool                      bFadeShineStrength;
  tAnimation<MappedWater>*  reflStrengthFader;
  float                     newReflStrength;
  float                     reflStrengthFadeTime;
  bool                      bFadeReflStrength;
  tAnimation<MappedWater>*  refractionFader;
  float                     newRefraction;
  float                     refractionFadeTime;
  bool                      bFadeRefraction;
  tAnimation<MappedWater>*  waterHeightFader;
  float                     newWaterHeight;
  float                     waterHeightFadeTime;
  bool                      bFadeWaterHeight;
  tAnimation<MappedWater>*  waterColorRFader;
  tAnimation<MappedWater>*  waterColorGFader;
  tAnimation<MappedWater>*  waterColorBFader;
  Vector                    newWaterColor;
  float                     waterColorFadeTime;
  bool                      bFadeWaterColor;
};

#endif
