/*!
 * @file md2Model.h
  *  Definition of an MD2 Model, a model format invented by ID Software.

    We are deeply thankfull for all the wunderfull things id software made for us gamers!

    The md2 file format is structured in a very simple way: it contains animations which are made out of
    frames (so called keyframes). Each frame is a complete draweable model in a specific position.
    A frame is a collection of vertex and its compagnions (normals, texture coordinates).

    A typical model has about 200 frames, the maximum frame count is fixed by MD2_MAX_FRAMES to currently
    512 frames. The maximal vetices count is set to 2048 verteces, not enough?
    You just have to change the MD2_MAX_* values if it doesn't fit your purposes...

    Surface Culling is fully implemented in md2 models: quake2 uses front face culling.
*/

#ifndef _MD2MODEL_H
#define _MD2MODEL_H

#include "base_object.h"

#include "interactive_model.h"
#include "material.h"

#include "md_model_structure.h"

//! These are the needed defines for the max values when loading .MD2 files
#define MD2_IDENT                       (('2'<<24) + ('P'<<16) + ('D'<<8) + 'I') //!< the md2 identifier tag in the bin file
#define MD2_VERSION                     8                                        //!< the md2 version in the header
#define MD2_MAX_TRIANGLES               4096                                     //!< maximal triangles count
#define MD2_MAX_VERTICES                2048                                     //!< maximal vertices count
#define MD2_MAX_TEXCOORDS               2048                                     //!< maximal tex coordinates
#define MD2_MAX_FRAMES                  512                                      //!< maximal frames
#define MD2_MAX_SKINS                   32                                       //!< maximal skins
#define MD2_MAX_FRAMESIZE               (MD2_MAX_VERTICES * 4 + 128)             //!< maximal framesize

#define NUM_VERTEX_NORMALS              162                                      //!< number of vertex normals
#define SHADEDOT_QUANT                  16                                       //!< shade dot quantity - no idea what it is

//! This stores the speed of the animation between each key frame - currently conflicting with the animation framework
#define kAnimationSpeed                 12.0f                                    //!< animation speed

//! This holds the header information that is read in at the beginning of the file: id software definition
struct MD2Header
{
   int ident;                           //!< This is used to identify the file
   int version;                         //!< The version number of the file (Must be 8)

   int skinWidth;                       //!< The skin width in pixels
   int skinHeight;                      //!< The skin height in pixels
   int frameSize;                       //!< The size in bytes the frames are

   int numSkins;                        //!< The number of skins associated with the model
   int numVertices;                     //!< The number of vertices (constant for each frame)
   int numTexCoords;                    //!< The number of texture coordinates
   int numTriangles;                    //!< The number of faces (polygons)
   int numGlCommands;                   //!< The number of gl commands
   int numFrames;                       //!< The number of animation frames

   int offsetSkins;                     //!< The offset in the file for the skin data
   int offsetTexCoords;                 //!< The offset in the file for the texture data
   int offsetTriangles;                 //!< The offset in the file for the face data
   int offsetFrames;                    //!< The offset in the file for the frames data
   int offsetGlCommands;                //!< The offset in the file for the gl commands data
   int offsetEnd;                       //!< The end of the file offset
};


//! animation names enumeration
typedef enum MD2animType
  {
    STAND,                       //0
    RUN,                         //1
    ATTACK,                      //2
    PAIN_A,                      //3
    PAIN_B,                      //4
    PAIN_C,          //5
    JUMP,            //6
    FLIP,            //7
    SALUTE,          //8
    FALLBACK,        //9
    WAVE,            //10
    POINT,           //11
    CROUCH_STAND,
    CROUCH_WALK,
    CROUCH_ATTACK,
    CROUCH_PAIN,
    CROUCH_DEATH,
    DEATH_FALLBACK,
    DEATH_FALLFORWARD,
    DEATH_FALLBACKSLOW,
    BOOM,

    MAX_ANIMATIONS
  };


typedef enum animPlayback
{
  MD2_ANIM_LOOP = 0,
  MD2_ANIM_ONCE,

  MD2_ANIM_NUM
};



/* forward definitions */
class Material;



//! class to store the md2 data in
class MD2Data : public BaseObject
{
 public:
   MD2Data(const std::string& modelFileName, const std::string& skinFileName, float scale = 1.0f);
  virtual ~MD2Data();

 private:
   bool loadModel(const std::string& fileName);
   bool loadSkin(const std::string& fileName = "");

 public:
  int                numFrames;             //!< number of frames
  int                numVertices;           //!< number of vertices
  int                numTriangles;          //!< number of triangles
  int                numTexCoor;            //!< number of texture coordinates
  int                numGLCommands;         //!< number of gl commands in the glList (tells how to draw)
  std::string        fileName;              //!< file name of the model File
  std::string        skinFileName;          //!< file name of the skin file
  MD2Header*         header;                //!< the header file

  sVec3D*            pVertices;             //!< pointer to the vertices data block
  sTriangle*         pTriangles;            //!< pointer to the triangles data
  sTexCoor*          pTexCoor;              //!< pointer to the texture coordinate data
  int*               pGLCommands;           //!< pointer to the gllist data block
  int*               pLightNormals;         //!< pointer to the light normals

  Material           material;              //!< pointer to the material
  float              scaleFactor;           //!< the scale factor of the model, (global)
};




//! This is a MD2 Model class
class MD2Model : public InteractiveModel {

public:
  MD2Model(const std::string& modelFileName, const std::string& skinFileName = "", float scale = 1.0f);
  virtual ~MD2Model();

  virtual void draw() const;
  void renderFrameTriangles() const;

  virtual void setAnimation(int type, int animPlayback = MD2_ANIM_LOOP);
  virtual void setAnimation(int firstFrame, int lastFrame, int fps, int bStoppable, int animPlayback);
  /**  returns the current animation @returns animation type */
  inline int MD2Model::getAnimation() { return this->animationState.type; }
  virtual void setAnimationSpeed(float speed) { this->animationSpeed = speed; }
  virtual bool isAnimationFinished() { return (this->animationState.currentFrame == this->animationState.endFrame )?true:false; }
  /**  scales the current model @param scaleFactor: the factor [0..1] to use for scaling */
  void scaleModel(float scaleFactor) { this->scaleFactor = scaleFactor;}

  virtual void tick(float dtS);
  void debug();


private:
  void animate(float time);
  void processLighting();
  void interpolate(/*sVec3D* verticesList*/);
  void renderFrame() const ;


 public:
  /* these variables are static, because they are all the same for every model */
  static sVec3D       anorms[NUM_VERTEX_NORMALS];       //!< the anormals
  static float        anormsDots[SHADEDOT_QUANT][256];  //!< the anormals dot products
  static sAnim        animationList[21];                //!< the anomation list
   //! again one of these strange id software parts
  float*              shadeDots;

  MD2Data*            data;                             //!< the md2 data pointer

 private:
  float               scaleFactor;                      //!< the scale factor (individual)
  float               animationSpeed;                   //!< the speed of the animation (factor for the time)
  sAnimState          animationState;                   //!< animation state of the model
  sVec3D              verticesList[MD2_MAX_VERTICES];   //!< place to temp sav the vert
};





#endif /* _MD2MODEL_H */
