/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "limited_width_text.h"
#include "font.h"

/**
 * @brief creates a new Text Element
 * @param fontFile the Font to render this text in
 * @param type The renderType to display this font in
 */
LimitedWidthText::LimitedWidthText(const std::string& fontFile, unsigned int textSize, float lineWidth, DotsPosition dotsPosition)
    : Text(fontFile, textSize)
{
  this->setClassID(CL_LIMITED_WIDTH_TEXT, "LimitedWidthText");

  this->_dotsPosition = End;
  this->setLineWidth(lineWidth);
}

/**
 * @brief sets the maximum Line width
 * @param lineWidth the maximum lineWidth.
 */
void LimitedWidthText::setLineWidth(float lineWidth)
{
  this->_lineWidth = lineWidth;
  this->setupTextWidth();
}

/**
 * @brief sets the Dots Position
 * @param dotsPosition the Position of the Dots
 */
void LimitedWidthText::setDotsPosition(DotsPosition dotsPosition)
{
  this->_dotsPosition = dotsPosition;
  this->setupTextWidth();
}


/**
 * @brief draws the Text
 */
void LimitedWidthText::draw() const
{
  if (unlikely(this->_dotedText.empty()))
    return;
  glPushMatrix();
  glPushAttrib(GL_ENABLE_BIT);
  // transform for alignment.
  if (this->getAlignment() == TEXT_ALIGN_RIGHT)
    glTranslatef(-this->getSizeX2D(), 0, 0);
  else if (this->getAlignment() == TEXT_ALIGN_CENTER || this->getAlignment() == TEXT_ALIGN_SCREEN_CENTER)
    glTranslatef(-this->getSizeX2D()/2, 0, 0);

  // drawing this Text.
  this->font().select();

  glTranslatef(getAbsCoor2D().x, getAbsCoor2D().y, 0);
  glRotatef(this->getAbsDir2D(), 0, 0, 1);

  Glyph* tmpGlyph;
  float posX = 0.0f;
  glBegin(GL_QUADS);
  for (unsigned int i = 0; i < this->_dotedText.size(); i++)
  {
    if(likely((tmpGlyph = this->font().getGlyphArray()[this->_dotedText[i]]) != NULL))
    {
      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->maxX*this->size(), 0);

      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->maxX*this->size(), this->size());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->minX*this->size(), this->size());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->minX*this->size(), 0);

      posX += tmpGlyph->advance * this->size();
    }
  }
  glEnd();
  glPopAttrib();
  glPopMatrix();
}


/**
 * @brief setting up the Text-Width if DYNAMIC
 */
void LimitedWidthText::setupTextWidth()
{
  float dotsSize = this->font().getGlyphArray()[46]->advance * 3.0;

  float width = 0.0f;
  float maxWidth = this->_lineWidth / this->size();

  this->_dotedText = this->text();

  switch (this->_dotsPosition)
  {
    case End:
      for (unsigned int i = 0; i < this->text().size(); i++)
      {
        if (width + dotsSize > maxWidth )
        {
          this->_dotedText = this->text().substr(0, i) + "...";
          if (i > 0)
            width -= this->font().getGlyphArray()[this->text()[i-1]]->advance;
          width += dotsSize;
          break;
        }
        // Advance the Text.
        if(this->font().getGlyphArray()[this->text()[i]] != NULL)
          width += this->font().getGlyphArray()[this->text()[i]]->advance;
      }
      break;

    case Begin:
      int i = text().size() -1;
      for (; i >= 0; --i)
      {
        if (width + dotsSize > maxWidth )
        {
          this->_dotedText = std::string("...") + this->text().substr(i);
          if (i + 1 < (int)text().size() )
            width -= this->font().getGlyphArray()[this->text()[i+1]]->advance;
          width += dotsSize;
          break;
        }
        // Advance the Text.
        if(this->font().getGlyphArray()[this->text()[i]] != NULL)
          width += this->font().getGlyphArray()[this->text()[i]]->advance;
      }
      break;
  }
  this->setSizeX2D(width * this->size());
}

/**
 * @brief print out some nice debug output
 */
void LimitedWidthText::debug() const
{
  printf("Debug %s::%s \n", this->getClassName(), this->getName() );
}
