/*!
 * @file world_entity.h
 * Definition of the basic WorldEntity
 */

#ifndef _WORLD_ENTITY_H
#define _WORLD_ENTITY_H

#include "p_node.h"
#include "synchronizeable.h"
#include "model.h"

#include "cr_engine.h"
#include "object_manager.h"
#include "glincl.h"
#include <vector>

#include "aabb_tree_node.h"

#include "physics_interface.h"



// FORWARD DECLARATION
namespace OrxSound { class SoundBuffer; class SoundSource; }
namespace OrxGui { class GLGuiWidget; class GLGuiBar; };

class BVTree;
class BoundingVolume;
class AABBTreeNode;
class Model;
class CollisionHandle;
class Collision;


//class CharacterAttributes;


//! Basis-class all interactive stuff in the world is derived from
class WorldEntity : public PNode
{
public:
  WorldEntity();
  virtual ~WorldEntity ();

  virtual void loadParams(const TiXmlElement* root);

  void loadModel(const std::string& fileName, float scaling = 1.0f, unsigned int modelNumber = 0, unsigned int obbTreeDepth = 4);
  void setModel(Model* model, unsigned int modelNumber = 0);
  Model* getModel(unsigned int modelNumber = 0) const { return (this->models.size() > modelNumber)? this->models[modelNumber] : NULL; };

  inline void loadMD2Texture(const std::string& fileName) { this->md2TextureFileName = fileName; }

  /** @param visibility if the Entity should be visible (been draw) */
  void setVisibiliy (bool visibility) { this->bVisible = visibility; };
  /** @returns true if the entity is visible, false otherwise */
  inline bool isVisible() const { return this->bVisible; };

  virtual void reset();

  virtual void postSpawn ();
  virtual void leaveWorld ();

  virtual void tick (float time);
  virtual void draw () const;

  /* --- Collision Detection Block  --- */
  bool buildObbTree(int depth);
  virtual void collidesWith (WorldEntity* entity, const Vector& location);
  virtual void collidesWithGround(const Vector& location);
  virtual void collidesWithGround(const Vector& feet, const Vector& ray_1, const Vector& ray_2);


/** @returns a reference to the obb tree of this worldentity */
  inline BVTree* getOBBTree() const { return this->obbTree; };
  void drawBVTree(int depth, int drawMode) const;
  inline AABB* getModelAABB() { return (this->aabbNode)?this->aabbNode->getAABB():NULL;}

  /* --- Collision Reaction Block --- */
  void subscribeReaction(CREngine::CRType type);
  void subscribeReaction(CREngine::CRType type, long target1);
  void subscribeReaction(CREngine::CRType type, long target1, long target2);
  void subscribeReaction(CREngine::CRType type, long target1, long target2, long target3);
  void subscribeReaction(CREngine::CRType type, long target1, long target2, long target3, long target4);

  void unsubscribeReaction(CREngine::CRType type);
  void unsubscribeReaction();

  bool registerCollision(WorldEntity* entityA, WorldEntity* entityB, BoundingVolume* bvA, BoundingVolume* bvB);
  bool registerCollision(int type, WorldEntity* entity, WorldEntity* groundEntity, Vector normal, Vector position, bool bInWall = false);
  /** @return true if there is at least on collision reaction subscribed */
  inline bool isReactive() const { return this->bReactive; }

  CollisionHandle* getCollisionHandle(CREngine::CRType type) const { return this->collisionHandles[type]; }

  /** @returns true if this entity is standing on ground (BSP model) */
  bool isOnGround() const { return this->isOnGround(); }
  /** @param flag: marks if this entity is standing on ground */
  void setOnGround(bool flag) { this->bOnGround = flag; }

  virtual void hit(float damage);
  virtual void destroy();


  /* @returns the Count of Faces on this WorldEntity */
  //unsigned int getFaceCount () const { return (this->model != NULL)?this->model->getFaceCount():0; };
  //  void addAbility(Ability* ability);
  //  void removeAbility(Ability* ability);
  //  void setCharacterAttributes(CharacterAttributes* charAttr);
  //  CharacterAttributes* getCharacterAttributes();

  /* --- Object Manager Block --- */
  void toList(OM_LIST list);

  void toReflectionList();
  void removeFromReflectionList();

  /** @returns a Reference to the objectListNumber to set. */
  OM_LIST& getOMListNumber() { return this->objectListNumber; }
  /** @returns a Reference to the Iterator */
  ObjectManager::EntityList::iterator& getEntityIterator() { return this->objectListIterator; }

  void hide() { if( this->objectListNumber != OM_DEAD) this->lastObjectListNumber = this->objectListNumber; this->toList(OM_DEAD); }
  void unhide() { if( this->objectListNumber != this->lastObjectListNumber) this->toList(this->lastObjectListNumber); }


  /* --- Character Attribute Block --- */
  /** @returns the damage dealt by this world entity */
  float getDamage() const { return this->damage; }
  /** sets the damage dealt to @param damage damage per second */
  void setDamage(float damage) { this->damage = damage; }
  /** @returns the Energy of the entity */
  float getHealth() const { return this->health; };
  /** @returns the Maximum energy this entity can be charged with */
  float getHealthMax() const { return this->healthMax; }
  float increaseHealth(float health);
  float decreaseHealth(float health);
  void increaseHealthMax(float increaseHealth);
  OrxGui::GLGuiWidget* getHealthWidget();
  bool hasHealthWidget() const { return this->healthWidget; };

  virtual void varChangeHandler( std::list<int> & id );


  //FIXME: the PhysicsInterface and the whole PEngine should probably be reviewed. Here its just used to store the vars
  /* --- Physics Interface --- */
  inline PhysicsInterface getPhysicsInterface() const { return this->physicsInterface; }
  inline float getMass() const { return this->physicsInterface.getMass(); }
  inline float getTotalMass() const { return this->physicsInterface.getTotalMass(); }
  inline void setVelocity(const Vector& vel) { this->velocity = vel; }


  /* --- Misc Stuff Block --- */
  void debugWE() { this->debugEntity(); }
  ;  ///FIXME
  void debugEntity() const;


protected:
  void setHealth(float health) { this->health = health; this->updateHealthWidget();};
  void setHealthWidgetVisibilit(bool visibility);
  void setHealthMax(float healthMax);
  void createHealthWidget();

  //  CharacterAttributes*    charAttr;         //!< the character attributes of a world_entity
private:
  void updateHealthWidget();

private:
  /// TODO maybe we will move the following three entries and the corresponding functions to Playable AND NPC
  float                   damage;             //!< the damage dealt to other objects by colliding.
  float                   health;             //!< The Energy of this Entity, if the Entity has any energy at all.
  float                   healthMax;          //!< The Maximal energy this entity can take.
  OrxGui::GLGuiBar*       healthWidget;       //!< The Slider (if wanted).

  std::vector<Model*>     models;             //!< The model that should be loaded for this entity.
  std::string             md2TextureFileName; //!< the file name of the md2 model texture, only if this
  std::string             modelLODName;       //!< the name of the model lod file
  BVTree*                 obbTree;            //!< this is the obb tree reference needed for collision detection
  AABBTreeNode*           aabbNode;           //!< the tree node of the first level of a axis aligned bounding boxes tree: model dimension

  bool                    bCollide;           //!< If it should be considered for the collisiontest.
  bool                    bVisible;           //!< If it should be visible.

  OM_LIST                 objectListNumber;                //!< The ObjectList from ObjectManager this Entity is in.
  ObjectManager::EntityList::iterator objectListIterator;  //!< The iterator position of this Entity in the given list of the ObjectManager.
  OM_LIST                 lastObjectListNumber;            //!< the last ObjectList from the ObjectManager this Entity was is in



  float                   scaling;                         //!< model's scaling factor
  int                     scaling_handle;                  //!< handle for syncing var

  std::string             modelFileName;                   //!< model's file name
  int                     modelFileName_handle;            //!< handle for syncing var

  CollisionHandle*        collisionHandles[CREngine::CR_NUMBER];  //!< the list of the collision reactions
  bool                    bReactive;                              //!< true if there is at least one collision reaction subscibed

  PhysicsInterface        physicsInterface;                //!< the physics object of the WorldEntity
  bool                    bOnGround;                       //!< true if this entity is standing on the ground

  protected:
  Vector                  velocity;                        //!< speed of the entity

};

#endif /* _WORLD_ENTITY_H */
