/*!
 * @file font.h
 * brief Definition of the FONT-loading class
 *
 * !! IMPORTANT !! When using ttf fonts clear the license issues prior to
 * adding them to orxonox. This is really important, because we do not want
 * to offend anyone.
 */

#ifndef _FONT_H
#define _FONT_H

#include "material.h"

#include "font_data.h"


//! A class to handle a Font of a certain ttf-File/image-file, Size.
class Font : public Material /* TODO Material it should be */
{

public:
  Font();
  Font(const std::string& fontFile, unsigned int renderSize);
  Font(const std::string& imageFile);
  Font(char** xpmArray);
  Font(const Font& font);
  virtual ~Font();

  Font& operator=(const Font& font);
  /** @brief compare two fonts @param font the comparator, @returns true if they match */
  bool operator==(const Font& font) const { return this->data == font.data; };


  /// LOADING new Fonts
  bool loadFontFromTTF(const std::string& fontFile, unsigned int renderSize);
  bool loadFontFromSDL_Surface(SDL_Surface* surface);

  void setStyle(const std::string& renderStyle);

  /** @returns a Pointer to the Array of Glyphs */
  inline Glyph** getGlyphArray() const { return this->data->getGlyphArray(); };
  /** @returns the a pointer to the TTF */
  inline TTF_Font* getTTF() const { return this->data->getTTF(); };

  int getMaxHeight() const;
  int getMaxAscent() const;
  int getMaxDescent() const;

  void createAsciiImage(const std::string& fileName, unsigned int size) const;

  void debug() const;

private:
  void init();
  void initGlyphs(Uint16 from, Uint16 count);
  bool getGlyphMetrics(Glyph* glyph, Uint16 character);
  static void initDefaultFont();

  int findOptimalFastTextureSize();
  bool createFastTexture();

  bool setTexture(GLuint texture);


private:
  FontDataPointer           data;                //!< A Data-Pointer to a Font.

  static FontDataPointer    defaultFontData;     //!< a default font, that is used, if other fonts were unable to be loaded.
};

#endif /* _FONT_H */
