/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "font.h"

#ifdef HAVE_SDL_IMAGE_H
#include <SDL_image.h>
#else
#include <SDL/SDL_image.h>
#endif

#include "default_font.xpm"

#include "debug.h"
#include "compiler.h"


Font::Font()
  : data(Font::defaultFontData)
{
  this->init();

}

/**
 * @brief constructs a Font out of a TTF-FIle
 * @param fontFile the File to load the font from
 * @param fontSize the Size of the Font in Pixels
 */
Font::Font(const std::string& fontFile, unsigned int renderSize)
  : data(Font::defaultFontData)
{
  this->init();

  if (!fontFile.empty())
    this->loadFontFromTTF(fontFile, renderSize);
}


/**
 * @brief constructs a Font out of an ImageFile
 * @param imageFile the ImageFile to load the Font From.
 */
Font::Font(const std::string& imageFile)
  : data(Font::defaultFontData)
{
  this->init();

  this->setName(imageFile);
  //  this->setSize(fontSize);
  SDL_Surface* image = NULL;
  if (!imageFile.empty())
    image = IMG_Load(imageFile.c_str());
  else
    return;
  if (image != NULL)
  {
    this->loadFontFromSDL_Surface(image);
    SDL_FreeSurface(image);
  }
  else
    PRINTF(1)("loading from surface %s failed: %s\n", imageFile.c_str(), IMG_GetError());
}

/**
 * @brief constructs a Font
 * @param xpmArray the xpm-ARRAY to load the font from
 */
Font::Font(char** xpmArray)
  : data(Font::defaultFontData)
{
  this->init();
  this->setName("XPM-array-font");
  //  this->setSize(fontSize);
  SDL_Surface* image = NULL;
  if (xpmArray != NULL)
    image = IMG_ReadXPMFromArray(xpmArray);
  if (image != NULL)
  {
    this->loadFontFromSDL_Surface(image);
    SDL_FreeSurface(image);
  }
  else
    PRINTF(1)("Loading from XPM-array failed: %s\n", IMG_GetError());
}


/**
 * @brief destructs a font
 *
 * this releases the memory a font uses to be opened.
 * deletes the glLists, and the TTF-handler, if present.
 */
Font::~Font()
{ }

Font& Font::operator=(const Font& font)
{
  Material::operator=(font);
  this->data = font.data;

  return *this;
};


/**
 * @brief initializes a Font (with default values)
 */
void Font::init()
{
  this->setBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  this->setClassID(CL_FONT, "Font");
  if (Font::defaultFontData.get() == NULL)
  {
     Font::initDefaultFont();
    this->data = Font::defaultFontData;
  }
}

FontDataPointer Font::defaultFontData(NULL);

/**
 * @brief initializes the default font
 */
void Font::initDefaultFont()
{
  // temporarily create a Font.
  Font::defaultFontData = FontDataPointer(new FontData);
  // apply the Data.
  printf("before: %p\n", defaultFontData.get());
  Font::defaultFontData = Font(font_xpm).data;
  printf("after: %p\n", defaultFontData.get());
}

/**
 * @brief sets The Font.
 * @param fontFile The file containing the font.
 * @returns true if loaded, false if something went wrong, or if a font was loaded before.
 */
bool Font::loadFontFromTTF(const std::string& fontFile, unsigned int renderSize)
{
  this->data = FontDataPointer (new FontData());

  this->setName(fontFile);
  this->data->fontTTF = TTF_OpenFont(fontFile.c_str(), renderSize);
  this->data->renderSize = renderSize;

  if(this->data->fontTTF != NULL)
  {
    this->setStyle("c");
    if (this->createFastTexture())
      return true;
    else
    {
      this->data = Font::defaultFontData;
      return false;
    }
  }
  else
  {
    PRINTF(1)("TTF_OpenFont: %s\n", TTF_GetError());
    this->data = Font::defaultFontData;
    return false;
  }

}

/**
 * @brief loads a font From an XPM-array.
 * @param xpmArray the array of the XPM to load the font from.
 */
bool Font::loadFontFromSDL_Surface(SDL_Surface* surface)
{
  if(surface == NULL)
    return false;

  this->data = FontDataPointer (new FontData());
  // loading to a texture.

  bool hasAlpha;
  SDL_Surface* newSurf = Texture::prepareSurface(surface, hasAlpha);
  if (newSurf != NULL)
  {
    this->data->texData->setSurface(newSurf);
    this->data->texData->setAlpha(hasAlpha);
    this->setTexture(Texture::loadTexToGL(newSurf));
  }
  else
  {
    this->data = Font::defaultFontData;
    return false;
  }

  // initializing the Glyphs.
  if (this->data->glyphArray == NULL)
  {
    float cx,cy;
    Glyph* tmpGlyph;
    this->data->glyphArray = new Glyph*[FONT_HIGHEST_KNOWN_CHAR];
    for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
    {
      tmpGlyph = this->data->glyphArray[i] = new Glyph;
      cx=(float)(i%16)/16.0f;                  // X Position Of Current Character
      cy=(float)(i/16)/16.0f;                  // Y Position Of Current Character

      tmpGlyph->texCoord[0] = cx;
      tmpGlyph->texCoord[1] = cx+0.0625f;
      tmpGlyph->texCoord[2] = cy;
      tmpGlyph->texCoord[3] = cy+0.0625f;
      tmpGlyph->minX = 0.0f;
      tmpGlyph->maxX = 1.0f;
      tmpGlyph->minY = 0.0f;
      tmpGlyph->maxY = 1.0f;
      tmpGlyph->width = 1.0f;
      tmpGlyph->advance = 1.0f;
      tmpGlyph->bearingX = 0.0f;
      tmpGlyph->bearingY = 0.0f;
      tmpGlyph->height = 1.0f;
    }
  }
  return true;
}


/**
 * @brief sets a specific data->renderStyle
 * @param data->renderStyle the Style to render: a string (char-array) containing:
 *   i: italic, b: bold, u, underline
 */
void Font::setStyle(const std::string& renderStyle)
{
  this->data->renderStyle = TTF_STYLE_NORMAL;

  for (unsigned int i = 0; i < renderStyle.size(); i++)
  {
    if (renderStyle[i] == 'b')
      this->data->renderStyle |= TTF_STYLE_BOLD;
    else if (renderStyle[i] == 'i')
      this->data->renderStyle |= TTF_STYLE_ITALIC;
    else if (renderStyle[i] == 'u')
      this->data->renderStyle |= TTF_STYLE_UNDERLINE;
  }
  if (likely(this->data->fontTTF != NULL))
    TTF_SetFontStyle(this->data->fontTTF, this->data->renderStyle);
  //  else
  //    PRINTF(2)("Font was not initialized, please do so before setting the Font-Style.\n");
}

/**
 * @brief creates and exports an Image, that has all the characters
 * stored in a Array (as an image)
 * @param fileName the File to write the image into.
 */
void Font::createAsciiImage(const std::string& fileName, unsigned int size) const
{
  if (this->data->fontTTF == NULL)
    return;
  int height = this->getMaxHeight();

  //
  // Surface definition.
  SDL_Rect tmpRect; // this represents a Rectangle for blitting.
  SDL_Surface* tmpSurf =  SDL_CreateRGBSurface(SDL_SWSURFACE,
                          height*size, height*size,
                          32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                          0x000000FF,
                          0x0000FF00,
                          0x00FF0000,
                          0xFF000000
#else
                          0xFF000000,
                          0x00FF0000,
                          0x0000FF00,
                          0x000000FF
#endif
                                              );
  tmpRect.x = 0; tmpRect.y = 0; tmpRect.w = tmpSurf->w; tmpRect.h = tmpSurf->h;
  SDL_SetClipRect(tmpSurf, &tmpRect);

  int posX, posY;
  // all the interessting Glyphs
  for (posY = 0; posY < 16; posY++)
  {
    for (posX = 0; posX < 16; posX++)
    {
      SDL_Surface* glyphSurf = NULL;
      if (likely(this->data->fontTTF != NULL))
      {
        SDL_Color white = {255, 255, 255};
        glyphSurf = TTF_RenderGlyph_Blended(this->data->fontTTF, posX+size*posY, white);
      }
      if( glyphSurf != NULL )
      {
        tmpRect.x = height*posX;
        tmpRect.y = height*posY;
        SDL_SetAlpha(glyphSurf, 0, 0);

        SDL_BlitSurface(glyphSurf, NULL, tmpSurf, &tmpRect);
        SDL_FreeSurface(glyphSurf);
      }
    }
  }
  SDL_SaveBMP(tmpSurf, fileName.c_str());
  SDL_FreeSurface(tmpSurf);
}


/**
 * @returns the maximum height of the Font, if the font was initialized, 0 otherwise
 */
int Font::getMaxHeight() const
{
  if (likely (this->data->fontTTF != NULL))
    return TTF_FontHeight(this->data->fontTTF);
  else
    return 1;
}

/**
 * @returns the maximum ascent of the Font, if the font was initialized, 0 otherwise
 *
 * the ascent is the pixels of the font above the baseline
 */
int Font::getMaxAscent() const
{
  if (likely(this->data->fontTTF != NULL))
    return TTF_FontAscent(this->data->fontTTF);
  else
    return 0;
}

/**
 * @returns the maximum descent of the Font, if the font was initialized, 0 otherwise
 *
 * the descent is the pixels of the font below the baseline
 */
int Font::getMaxDescent() const
{
  if (likely(this->data->fontTTF != NULL))
    return TTF_FontDescent(this->data->fontTTF);
  else
    return 0;
}

/**
 * @param glyph: The Glyph to set the Parameters to.
 * @param character: The character to get info about.
 * @returns a Glyph struct of a character. This Glyph is a pointer,
 * and MUST be deleted by the user..
 *
 * This only works for horizontal fonts. see
 * http://freetype.sourceforge.net/freetype2/docs/tutorial/step2.html
 * for more info about vertical Fonts
 */
bool Font::getGlyphMetrics(Glyph* glyph, Uint16 character)
{
  glyph->character = character;
  if (likely (this->data->fontTTF!= NULL))
  {
    int miX, maX, miY, maY, adv;
    if (TTF_GlyphMetrics(this->data->fontTTF, glyph->character,
                     &miX, &maX,
                     &miY, &maY,
                     &adv) == -1)
      return false;
    glyph->minX = (float)miX / (float)this->data->renderSize;
    glyph->maxX = (float)maX / (float)this->data->renderSize;
    glyph->minY = (float)miY / (float)this->data->renderSize;
    glyph->maxY = (float)maY / (float)this->data->renderSize;
    glyph->advance = (float)adv / (float)this->data->renderSize;

    // Calculate the Rest.
    glyph->height = glyph->maxY - glyph->minY;
    glyph->width = glyph->maxX - glyph->minX;
    glyph->bearingX = (glyph->advance - glyph->width) / 2;
    glyph->bearingY = glyph->maxY;

    //printf("%c:: %d %d %d %d %d\n", character, miX, maX, miY, maY, adv);

    return true;
  }
  return false;
}

/**
 * @brief creates a Fast-Texture of this Font
 */
bool Font::createFastTexture()
{
  /* interesting GLYPHS:
  *  32: space
  *  33-47: Special Characters.
  *  48-57: 0-9
  *  58-63: some more special chars (minor)
  *  65-90: A-Z
  *  97-122: a-z
  */
  int numberOfGlyphs = 91;

  this->initGlyphs(32, numberOfGlyphs);
  //  this->data->glyphArray[32]->width = .5f; //!< @todo find out the real size of a Space

  int rectSize = this->findOptimalFastTextureSize();

  // setting default values. (maybe not needed afterwards)
  SDL_Color tmpColor;  tmpColor.r = tmpColor.g = tmpColor.b = 0;
  // Surface definition.
  SDL_Rect tmpRect; // this represents a Rectangle for blitting.
  SDL_Surface* tmpSurf =  SDL_CreateRGBSurface(SDL_HWSURFACE,
                          rectSize, rectSize,
                          32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                          0x000000FF,
                          0x0000FF00,
                          0x00FF0000,
                          0xFF000000
#else
                          0xFF000000,
                          0x00FF0000,
                          0x0000FF00,
                          0x000000FF
#endif
                                              );
  tmpRect.x = 0; tmpRect.y = 0; tmpRect.w = tmpSurf->w; tmpRect.h = tmpSurf->h;
  SDL_SetClipRect(tmpSurf, &tmpRect);
  int maxLineHeight = this->getMaxHeight();

  // all the interessting Glyphs
  for (int i = 0; i < 128; i++)
  {
    SDL_Surface* glyphSurf = NULL;
    Glyph* tmpGlyph;

    if (tmpGlyph = this->data->glyphArray[i])
    {
      if (tmpGlyph->height*this->data->renderSize > maxLineHeight)
        maxLineHeight = (int)(tmpGlyph->height*this->data->renderSize);

      if (tmpRect.x+tmpGlyph->advance*this->data->renderSize > tmpSurf->w)
      {
        tmpRect.x = 0;
        tmpRect.y = tmpRect.y + maxLineHeight;
      }
      if (tmpRect.y + maxLineHeight > tmpSurf->h)
      {
        PRINTF(1)("Protection, so font cannot write over the boundraries (!!this should not heappen!!)\n");
        break;
      }
      // reading in the new Glyph
      if (likely(this->data->fontTTF != NULL))
      {
        SDL_Color white = {255, 255, 255};
        glyphSurf = TTF_RenderGlyph_Blended(this->data->fontTTF, i, white);
      }
      if( glyphSurf != NULL )
      {
        SDL_SetAlpha(glyphSurf, 0, 0);
        int tmpY = tmpRect.y;
        tmpRect.y += this->getMaxAscent()-(int)((float)tmpGlyph->bearingY*this->data->renderSize);
        SDL_BlitSurface(glyphSurf, NULL, tmpSurf, &tmpRect);
        tmpRect.y = tmpY;

        tmpGlyph->texCoord[0] = (float)((float)tmpRect.x )/(float)tmpSurf->w;
        tmpGlyph->texCoord[1] = (float)((float)tmpRect.x + tmpGlyph->width*(float)this->data->renderSize)/(float)tmpSurf->w;
        tmpGlyph->texCoord[2] = (float)(tmpRect.y)/(float)tmpSurf->w;
        tmpGlyph->texCoord[3] = (float)((float)tmpRect.y+(float)this->getMaxHeight())/(float)tmpSurf->w;
        SDL_FreeSurface(glyphSurf);
        tmpRect.x += (int)(tmpGlyph->width * this->data->renderSize) + 1;
      }
    }
  }
  // outputting the GLYPH-table
//       char outName[1024];
//       sprintf( outName, "%s-glyphs.bmp", this->getName());
//       SDL_SaveBMP(tmpSurf, outName);
  this->data->texData->setAlpha(true);
  if (this->data->texData->setSurface(tmpSurf))
    this->setTexture(Texture::loadTexToGL(tmpSurf));
  return true;
}

/**
 * @brief the Internal implementation of setting up the Texture.
 * @param texture the Texture to load.
 * @returns true on success, false otherwise.
 */
bool Font::setTexture(GLuint texture)
{
  bool retVal = this->data->texData->setTexture(texture);
  this->setDiffuseMap(data->texData, 0);
  printf("this->texture %d\n", texture);
  //printf(this->getT)
  this->debug();
  return retVal;
};

/**
 * @brief stores Glyph Metrics in an Array.
 * @param from The Glyph to start from.
 * @param count The number of Glyphs to start From.
 */
void Font::initGlyphs(Uint16 from, Uint16 count)
{
  /* initialize the Array, and set all its entries to NULL
  *  only if the Glyph-array has not been initialized
  */
  if (!this->data->glyphArray)
  {
    this->data->glyphArray = new Glyph*[FONT_HIGHEST_KNOWN_CHAR];
    for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
      this->data->glyphArray[i] = NULL;
  }

  Uint16 lastGlyph = from + count;

  for (int i = from; i <= lastGlyph; i++)
  {
    // setting up all the Glyphs we like.
    Glyph* newGlyph = new Glyph;
    if (getGlyphMetrics(newGlyph, i))
      data->glyphArray[i] = newGlyph;
    else
    {
      delete newGlyph;
      data->glyphArray[i] = NULL;
    }
  }
  return;
}

/**
 * @returns the optimal size to use as the texture size
 *
 * @todo: this algorithm can be a lot faster, althought it does
 * not really matter within the init-context, and 128 glyphs.
 *
 * This function searches for a 2^n sizes texture-size, this is for
 * openGL-version < 1.2 compatibility ( and because it is realy easy like this :))
 */
int Font::findOptimalFastTextureSize()
{
  if (this->data->glyphArray == NULL)
    return 0;

  unsigned int i;
  unsigned int x,y; // the counters
  int maxLineHeight = this->getMaxHeight();
  unsigned int size = 32;  // starting Value, we have to start somewhere 32 seems reasonable. (take any small enough 2^i number)
  bool sizeOK = false;
  Glyph* tmpGlyph;

  while (!sizeOK)
  {
    x = 0; y = 0;
    for (i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
    {
      if((tmpGlyph = this->data->glyphArray[i]) != NULL)
      {
        // getting the height of the highest Glyph in the Line.
        if (tmpGlyph->height*this->data->renderSize > maxLineHeight)
          maxLineHeight = (int)(tmpGlyph->height*this->data->renderSize);

        if (x + tmpGlyph->advance*this->data->renderSize > size)
        {
          x = 0;
          y = y + maxLineHeight;
          //maxLineHeight = 0;
        }
        if (y + maxLineHeight + 1 > size)
          break;
        x += (int)(tmpGlyph->advance*this->data->renderSize)+1;

      }
    }
    if (i >= FONT_HIGHEST_KNOWN_CHAR-1 || size > 8192)
      sizeOK = true;
    else
      size *= 2;
  }
  return size;
}


/**
 * @brief a simple function to get some interesting information about this class
 */
void Font::debug() const
{
  Material::debug();

  PRINT(0)("TEST %p and %p\n", this->data.get(), this->data->texData.get());
  // print the loaded font's style
  int style = TTF_STYLE_NORMAL;
  if (likely(this->data->fontTTF != NULL))
    style = TTF_GetFontStyle(this->data->fontTTF);
  PRINTF(0)("The font style is:");
  if(style==TTF_STYLE_NORMAL)
    PRINTF(0)(" normal");
  else
  {
    if(style&TTF_STYLE_BOLD)
      PRINTF(0)(" bold");
    if(style&TTF_STYLE_ITALIC)
      PRINTF(0)(" italic");
    if(style&TTF_STYLE_UNDERLINE)
      PRINTF(0)(" underline");
  }
  PRINTF(0)("\n");
}
