/*!
 * @file texture.h
 * @brief Contains the texture class, that handles the reading of Images into Texutre-files.
 */

#ifndef _TEXTURE_H
#define _TEXTURE_H

#include "base_object.h"

#include "glincl.h"
#include "count_pointer.h"
#include "texture_data.h"


/* Forward Declaration */
struct SDL_Surface;

//! A Class, that reads in Textures from different fileformats.
class Texture : public BaseObject
{
public:
  Texture();
  Texture(const Texture& texture);
  Texture(GLenum target, unsigned int width, unsigned int height, unsigned int channels, GLenum type);
  Texture(const std::string& imageName, GLenum target = GL_TEXTURE_2D);
  Texture(SDL_Surface* surface, GLenum target = GL_TEXTURE_2D);

  Texture& operator=(const Texture& texture);
  Texture& operator=(const TextureDataPointer& textureDataPointer);

  virtual ~Texture();

  bool loadImage(const std::string& imageName, GLenum target = GL_TEXTURE_2D);
  bool loadSurface(SDL_Surface* surface, GLenum target = GL_TEXTURE_2D);
  virtual bool rebuild();

  /** @returns The textureID of this texture.  */
  inline GLuint getTexture() const { return this->data->getTexture(); };
  /** @returns true if texture has alpha, false otherwise */
  inline bool hasAlpha() const  { return this->data->hasAlpha(); }
  /** @returns the stored image of this Texture */
  const SDL_Surface* const getStoredImage() const { return this->data->getStoredImage(); };



  static void setTextureEnableState(bool texturesEnabled);
  /** @returns true if Textures are enabled */
  inline static bool getTextureEnableState() { return Texture::texturesEnabled; };
  // Utility functionality:
  static SDL_Surface* prepareSurface(SDL_Surface* input, bool& hasAlpha);
  static GLuint loadTexToGL (const SDL_Surface* surface, GLenum target = GL_TEXTURE_2D);

protected:
  bool setSurface(SDL_Surface* newSurface) { return this->data->setSurface(newSurface); };
  bool setAlpha(bool hasAlpha) { return this->data->setAlpha(hasAlpha); };
  bool setTexture(GLuint texture) { return this->data->setTexture(texture); };

private:
  void init();
  static void generateTexture(GLuint& texture, GLenum target);

private:
  TextureDataPointer            data;               //!< The TextureData
  GLclampf                      priority;           //!< the priority of the current texture (used for garphics cards with limited mem)

  static bool                   texturesEnabled;    //!< If the Textures are enabled.
};

#endif /* _TEXTURE_H */
