/*!
 * @file font.h
 * brief Definition of the FONT-loading class
 *
 * !! IMPORTANT !! When using ttf fonts clear the license issues prior to
 * adding them to orxonox. This is really important, because we do not want
 * to offend anyone.
 */

#ifndef _FONT_H
#define _FONT_H

#include "texture.h"

#include "font_data.h"


//! A class to handle a Font of a certain ttf-File/image-file, Size.
class Font : public Texture
{

public:
  Font(const std::string& fontFile,
       unsigned int renderSize);
  Font(const std::string& imageFile);
  Font(char** xpmArray);
  virtual ~Font();

  //  font
  bool loadFontFromTTF(const std::string& fontFile);
  bool loadFontFromSDL_Surface(SDL_Surface* surface);

  void setStyle(const std::string& renderStyle);

  /** @returns a Pointer to the Array of Glyphs */
  inline Glyph** getGlyphArray() const { return this->data->getGlyphArray(); };
  /** @returns the a pointer to the TTF */
  inline TTF_Font* getTTF() const { return this->data->getTTF(); };

  int getMaxHeight() const;
  int getMaxAscent() const;
  int getMaxDescent() const;

  /** @returns the default Font */
  inline static Font* getDefaultFont() { if (Font::defaultFont == NULL) initDefaultFont(); return Font::defaultFont; };

  void createAsciiImage(const std::string& fileName, unsigned int size) const;
  static void initDefaultFont();
  static void removeDefaultFont();


private:
  void init();
  bool getGlyphMetrics(Glyph* glyph, Uint16 character);

  bool createFastTexture();

  void initGlyphs(Uint16 from, Uint16 count);
  int findOptimalFastTextureSize();

  void debug();

private:
  static Font*  defaultFont;         //!< a default font, that is used, if other fonts were unable to be loaded.

  fontDataPointer data;
};

#endif /* _FONT_H */
