/*!
 * @file font_data.h
 * @brief Contains the font-data class, that handles the reading of Images into Texutre-files.
 */

#ifndef _FONT_DATA_H
#define _FONT_DATA_H

#include "base_object.h"

#include "glincl.h"
#include "texture_data.h"

#ifdef HAVE_SDL_TTF_H
#include <SDL_ttf.h>
#else
#include <SDL/SDL_ttf.h>
#endif



/* some default values */
#define FONT_NUM_COLORS              256           //!< number of colors.

#define FONT_HIGHEST_KNOWN_CHAR      128           //!< The highest character known to the textEngine.
#define FONT_DEFAULT_RENDER_SIZE     50            //!< At what Resolution to render fonts.

//! A struct for handling glyphs
/**
 * a Glyph is one letter of a certain font
 */
struct Glyph
{
  // Glyph-specific (size and so on)
  Uint16   character;         //!< The character
  float    minX;              //!< The minimum distance from the origin in X
  float    maxX;              //!< The maximum distance from the origin in X
  float    minY;              //!< The minimum distance from the origin in Y
  float    maxY;              //!< The maximum distance from the origin in Y
  float    width;             //!< The width of the Glyph
  float    height;            //!< The height of the Glyph
  float    bearingX;          //!< How much is right of the Origin
  float    bearingY;          //!< How much is above the Origin
  float    advance;           //!< How big a Glyph would be in monospace-mode

  GLfloat texCoord[4];        //!< Texture coordinates: 0:left, 1:right, 2: top, 3: bottom.
};


class FontData : public TextureData
{
  friend class Font;
public:
  FontData();
  ~FontData();


  /** @returns a Pointer to the Array of Glyphs */
  inline Glyph** getGlyphArray() const { return this->glyphArray; };
  /** @returns the a pointer to the TTF */
  inline TTF_Font* getTTF() const { return this->fontTTF; };

  bool rebuild();


private:
  TTF_Font*     fontTTF;             //!< The font we use for this.
  int           renderStyle;         //!< The Renderstyle
  unsigned int  renderSize;          //!< How big the Font should be rendered.

  Glyph**       glyphArray;          //!< An Array of all the Glyphs stored in the Array of Glyphs.
};

typedef CountPointer<FontData> fontDataPointer;

#endif /* _FONT_DATA_H */
