/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "md3_data.h"

#include "md3_bone_frame.h"
#include "md3_tag.h"
#include "md3_mesh.h"

#include "material.h"

#include "debug.h"

namespace md3
{

/********************************************************************************
 *   MD3Data                                                                    *
 ********************************************************************************/

/**
  \brief simple constructor
*/
MD3Data::MD3Data(const std::string& modelFileName, const std::string& skinFileName, float scale)
{
  this->filename = modelFileName;

  this->parentTagIndex = -1;
  this->parent = NULL;

  this->animationState.currentFrame = 0;
  this->animationState.nextFrame = 1;
  this->animationState.interpolationFraction = 0.0f;

  this->animation = NULL;
  this->bInterpolate = false;
  this->upperBound = 0;

  this->loadModel(modelFileName);
//   this->loadSkin(skinFileName);
  this->init();
}


/**
  \brief simple destructor

  this will clean out all the necessary data for a specific md2model
*/
MD3Data::~MD3Data()
{
  delete this->header;
}


/**
 * init data
 */
void MD3Data::init()
{
  // create the temporary data to work with (interpolation data)
  this->tmpBoneFrame = new MD3BoneFrame();

  this->tmpMesh = new sVec3D*[this->header->meshNum];
  for( int i = 0; i < this->header->meshNum; i++)
    this->tmpMesh[i] = new sVec3D[this->meshes[i]->header->vertexNum];

  this->tmpNormal = new MD3Normal*[this->header->meshNum];
  for( int i = 0; i < this->header->meshNum; i++)
    this->tmpNormal[i] = new MD3Normal[this->meshes[i]->header->vertexNum];

  // there are at most 4 different models and submodels
  this->tmpMatrix = new float*[4];
  for( int i = 0; i < 4; i++)
    this->tmpMatrix[i] = new float[16];

}


/**
 * link a model at the specified tag position to this model. if the position is already
 * occupied, the old submodel will be replaced
 *
 *  @param tagIndex: tag to link the submodel to
 *  @param child: the submodel that should be linked to this model
 */
void MD3Data::addLinkedModel(int tagIndex, MD3Data* child)
{
  this->sortedMap[tagIndex] = child;
  child->parentTagIndex = tagIndex;
}



/**
 * Return the index of the tag with the given name for this model.
 *
 * This will return -1 if their is no such tag.
 */
 int MD3Data::getTagIndexByName(std::string tagName)
{
   int res = -1;

   if( this->header->boneFrameNum > 0) {
     MD3Tag** tags = this->boneFrames[0]->tags;
     for( int i = 0; i < this->header->tagNum; i++)
       if( tags[i]->name.find(tagName) == std::string::npos)
         return i;
   }

   return res;
 }



/**
  \brief this will load the whole model data (vertices, opengl command list, ...)
* @param fileName: the name of the model file
  \return true if success
*/
bool MD3Data::loadModel(const std::string& fileName)
{
  FILE *pFile;                            //file stream
//  char* buffer;                           //buffer for frame data
  int fileOffset = 0;                     // file data offset


  //! @todo this chek should include deleting a loaded model (eventually)
  if (fileName.empty())
    return false;

  PRINTF(0)("opening file: %s\n", fileName.c_str());
  pFile = fopen(fileName.c_str(), "rb");
  if( unlikely(!pFile))
    {
      PRINTF(1)("Couldn't open the MD3 File for loading. Exiting.\n");
      return false;
    }
  fileOffset += this->readHeader(pFile, fileOffset);
  /* check for the header version: make sure its a md2 file :) */
  if( unlikely(this->header->version != MD3_VERSION) && unlikely(this->header->ident != MD3_IDENT))
    {
      PRINTF(1)("Couldn't load file %s: invalid file format: stop loading\n", fileName.c_str());
      return false;
    }


    // check if the filesize is correct
    if( this->header->fileSize > this->header->tagStart &&
        this->header->fileSize >= this->header->meshStart)
    {
      bool bBoneFrames, bTags, bMeshes;
      bBoneFrames = ( this->header->boneFrameNum == 0);
      bTags = ( this->header->tagNum == 0);
      bMeshes = ( this->header->meshNum == 0);

      // read different parts of the model in correct order
      while( !(bBoneFrames && bTags && bMeshes))
      {
        printf("while, fileOffset = %i\n", fileOffset);
        if( fileOffset == this->header->boneFrameStart && !bBoneFrames)
        {
          fileOffset += this->readBoneFrames(pFile, fileOffset);
          bBoneFrames = true;
        }
        else if( fileOffset == this->header->tagStart && !bTags)
        {
          fileOffset += this->readTags(pFile, fileOffset);
          bTags = true;
        }
        else if( fileOffset == this->header->meshStart && !bMeshes)
        {
          fileOffset += this->readMeshes(pFile, fileOffset);
          bMeshes = true;
        }
      }

    }

  fclose(pFile);

  return true;
}


/**
  \brief loads the skin/material stuff
* @param fileName: name of the skin file
  \return true if success
*/
bool MD3Data::loadSkin(const std::string& fileName)
{
//   if( fileName.empty())
//     {
//       this->skinFileName = "";
//       return false;
//     }
//
//   this->skinFileName = fileName;
//
//   this->material.setName("md2ModelMaterial");
//   this->material.setDiffuseMap(fileName);
//   this->material.setIllum(3);
//   this->material.setAmbient(1.0, 1.0, 1.0);

  return true;
}


/**
 * read heaader
 */
int MD3Data::readHeader(FILE* pFile, int fileOffset)
{
  PRINTF(0)("Reading Header\n");

  this->header = new MD3Header;
  fread(this->header, 1, sizeof(MD3Header), pFile);

    //header debug:
  PRINTF(0)("MD3 Header debug section======================================\n");
  printf("ident: %i\n", this->header->ident);
  printf("version: %i\n", this->header->version);
  printf("filename: %s\n", this->header->filename);
  printf("boneFrameNum: %i\n", this->header->boneFrameNum);
  printf("tag number: %i\n", this->header->tagNum);
  printf("mesh number: %i\n", this->header->meshNum);
  printf("max tex num: %i\n", this->header->maxTexNum);
  printf("bone frame start: %i\n", this->header->boneFrameStart);
  printf("tag start: %i\n", this->header->tagStart);
  printf("mesh start: %i\n", this->header->meshStart);
  printf("fileSize: %i\n", this->header->fileSize);

  return sizeof(MD3Header);
}


/**
 * read bone frames
 */
int MD3Data::readBoneFrames(FILE* pFile, int fileOffset)
{
  PRINTF(0)("Reading Bone Frames\n");

  this->boneFrames = new MD3BoneFrame*[this->header->boneFrameNum];

  for( int i = 0; i < this->header->boneFrameNum; i++)
  {
    this->boneFrames[i] = new MD3BoneFrame(i);

    MD3BoneFrameData* md = new MD3BoneFrameData;
    fread(md, 1, sizeof(MD3BoneFrameData), pFile);
    this->boneFrames[i]->data = md;
  }

  return this->header->boneFrameNum * sizeof(MD3BoneFrameData);
}


/**
 * read the tags
 */
int MD3Data::readTags(FILE* pFile, int fileOffset)
{
  PRINTF(0)("Reading Tags\n");

  for( int i = 0; i < this->header->boneFrameNum; i++)
  {
    this->boneFrames[i]->tags = new MD3Tag*[this->header->tagNum];

    for( int j = 0; j < this->header->tagNum; j++)
    {
      this->boneFrames[i]->tags[j] = new MD3Tag();
      MD3TagData* md = new MD3TagData;
      fread(md, 1, sizeof(MD3TagData), pFile);
      this->boneFrames[i]->tags[j]->data = md;

      this->boneFrames[i]->tags[j]->name = std::string(this->boneFrames[i]->tags[j]->data->name);
      this->boneFrames[i]->tags[j]->position = Vector( this->boneFrames[i]->tags[j]->data->position[0],
                                                       this->boneFrames[i]->tags[j]->data->position[1],
                                                       this->boneFrames[i]->tags[j]->data->position[2]);
      for( int k = 0; k < 3; k++)
        for( int l = 0; l < 3; l++)
          this->boneFrames[i]->tags[j]->matrix[k][l] = this->boneFrames[i]->tags[j]->data->matrix[k][l];

      //PRINTF(0)("Tag name: %s\n", this->boneFrames[i]->tags[j]->name.c_str());
    }
  }

  return this->header->boneFrameNum * this->header->tagNum * sizeof(MD3TagData);
}


/**
 * read meshes
 */
int MD3Data::readMeshes(FILE* pFile, int fileOffset)
{
  PRINTF(0)("Reading Mesh Data\n");

  fileOffset = 0;

  this->meshes = new MD3Mesh*[this->header->meshNum];

  for( int i = 0; i < this->header->meshNum; i++)
  {
    this->meshes[i] = new MD3Mesh();

    int localFileOffset = 0;
    bool   bTriangles, bTexVecs, bVertices, bTextures;            //!< the parts that have been read so far

    //start reading mesh data
    MD3MeshHeader* md = new MD3MeshHeader;
    fread(md, 1, sizeof(MD3MeshHeader), pFile);
    this->meshes[i]->header = md;
    localFileOffset += sizeof(MD3MeshHeader);

    PRINTF(0)("MD3 Mesh Header debug section\n");
    printf("ident: %i\n", md->id);
    printf("filename: %s\n", md->name);
    printf("meshFrameNum: %i\n", md->meshFrameNum);
    printf("textureNum: %i\n", md->textureNum);
    printf("vertexNum: %i \n", md->vertexNum);
    printf("triangleNum: %i\n", md->triangleNum);
    printf("triangleStart: %i\n", md->triangleStart);
    printf("textureStart: %i\n", md->textureStart);
    printf("texVecStart: %i\n", md->texVecStart);
    printf("vertexStart: %i\n", md->vertexStart);
    printf("fileSize: %i\n", md->meshSize);

    if( unlikely(this->meshes[i]->header->id != MD3_IDENT))
    {
      PRINTF(1)("Wrong MD3 mesh file tag, file %s could be corrupt\n", this->filename.c_str());
      return false;
    }

    // check which parts to be loaded
    bTriangles = ( this->meshes[i]->header->triangleNum == 0);
    bTexVecs = ( this->meshes[i]->header->vertexNum == 0);
    bVertices = ( this->meshes[i]->header->meshFrameNum == 0);
    bTextures = ( this->meshes[i]->header->textureNum == 0);

    // now read the data block whise
    while( !(bTriangles && bTexVecs && bVertices && bTextures))
    {
      PRINTF(0)("while2: localOffset = %i\n", localFileOffset);
      if( localFileOffset == this->meshes[i]->header->triangleStart  && !bTriangles)
      {
        localFileOffset += this->readMeshTriangles(pFile, localFileOffset, i);
        bTriangles = true;
      }
      else if( localFileOffset == this->meshes[i]->header->textureStart && !bTextures)
      {
        localFileOffset += this->readMeshTextures(pFile, localFileOffset, i);
        bTextures = true;
      }
      else if( localFileOffset == this->meshes[i]->header->texVecStart && !bTexVecs)
      {
        localFileOffset += this->readMeshTexVecs(pFile, localFileOffset, i);
        bTexVecs = true;
      }
      else if( localFileOffset == this->meshes[i]->header->vertexStart && !bVertices)
      {
        localFileOffset += this->readMeshVertices(pFile, localFileOffset, i);
        bVertices = true;
      }
    }
    fileOffset += localFileOffset;
    PRINTF(0)("finished reading mesh %i, got %i of %i byes\n", i, localFileOffset, md->meshSize);
  }
  return fileOffset;
}



/**
 * reading in the mesh triangles
 */
int MD3Data::readMeshTriangles(FILE* pFile, int fileOffset, int mesh)
{
  PRINTF(0)("Reading Mesh Triangles\n");
  // create the memomry to save the triangles
  this->meshes[mesh]->triangles = new MD3Triangle[this->meshes[mesh]->header->triangleNum];
  fread(this->meshes[mesh]->triangles, 1, sizeof(MD3Triangle) * this->meshes[mesh]->header->triangleNum, pFile);

//   for( int i = 0; i < this->meshes[mesh]->header->triangleNum; i++) {
//     PRINTF(0)("Triangle read: %i, %i, %i\n", this->meshes[mesh]->triangles[i].vertexOffset[0], this->meshes[mesh]->triangles[i].vertexOffset[1]
//         , this->meshes[mesh]->triangles[i].vertexOffset[2]);
//   }

  return this->meshes[mesh]->header->triangleNum * sizeof(MD3Triangle);
}


/**
 * reading in the mesh textures
 */
int MD3Data::readMeshTextures(FILE* pFile, int fileOffset, int mesh)
{
  PRINTF(0)("Reading Mesh Textures\n");

  // create the textures
  this->meshes[mesh]->material = new Material[this->meshes[mesh]->header->textureNum];

  MD3Texture* tex = new MD3Texture[this->meshes[mesh]->header->textureNum];
  fread(tex, 1, sizeof(MD3Texture) * this->meshes[mesh]->header->textureNum, pFile);

  for( int i = 0; i < this->meshes[mesh]->header->textureNum; i++) {
    PRINTF(0)(" texture file: %s\n", tex[i].fileName);
#warning texture stuff hard coded. make this again
    std::string path("/home/boenzlip/tmp/q3/Downloads/MOH/q3mdl-alien3/");
    std::string path1(tex[i].fileName);
    std::string fullPath( path + path1);
    this->meshes[mesh]->material[i].setDiffuseMap(/*tex[i].fileName*/ /*fullPath.c_str()*/ "maps/creatures/gork/gorkup.tga");
    this->meshes[mesh]->material[i].setAmbient(1, 1, 1);
  }

  return this->meshes[mesh]->header->textureNum * sizeof(MD3Texture);
}


/**
 * reading in the mesh tex vecs
 */
int MD3Data::readMeshTexVecs(FILE* pFile, int fileOffset, int mesh)
{
  PRINTF(0)("Reading Mesh TexVecs\n");

  this->meshes[mesh]->texVecs = new MD3TexVecs[this->meshes[mesh]->header->vertexNum];
  fread(this->meshes[mesh]->texVecs, 1, sizeof(MD3TexVecs) * this->meshes[mesh]->header->vertexNum, pFile);

//   for( int i = 0; i < this->meshes[mesh]->header->vertexNum; i++)  {
//     PRINTF(0)("TexVec read: %f, %f\n", this->meshes[mesh]->texVecs[i].textureCoord[0], this->meshes[mesh]->texVecs[i].textureCoord[0]);
//   }

  return this->meshes[mesh]->header->vertexNum * sizeof(MD3TexVecs);
}


/**
 * reading in the mesh vertices
 */
int MD3Data::readMeshVertices(FILE* pFile, int fileOffset, int mesh)
{
  PRINTF(0)("Reading Mesh Vertices\n");

  // reserver memory for the vertex informations
  this->meshes[mesh]->meshFrames = new sVec3D*[this->meshes[mesh]->header->meshFrameNum]; // * this->meshes[mesh]->header->vertexNum
  this->meshes[mesh]->normals = new MD3Normal*[this->meshes[mesh]->header->meshFrameNum];

  for( int i = 0; i < this->meshes[mesh]->header->meshFrameNum; i++)
  {
    this->meshes[mesh]->meshFrames[i] = new sVec3D[this->meshes[mesh]->header->vertexNum];
    this->meshes[mesh]->normals[i] = new MD3Normal[this->meshes[mesh]->header->vertexNum];

    for( int j = 0; j < this->meshes[mesh]->header->vertexNum; j++)
    {
      // read out the compressed data
      MD3VertexCompressed* vc = new MD3VertexCompressed;
      fread(vc, 1, sizeof(MD3VertexCompressed), pFile);

      this->meshes[mesh]->meshFrames[i][j][0] = (float)vc->vector[0] / 64.0f;
      this->meshes[mesh]->meshFrames[i][j][1] = (float)vc->vector[1] / 64.0f;
      this->meshes[mesh]->meshFrames[i][j][2] = (float)vc->vector[2] / 64.0f;

      this->meshes[mesh]->normals[i][j].vertexNormal[0] = (int)vc->vertexNormal[0];
      this->meshes[mesh]->normals[i][j].vertexNormal[1] = (int)vc->vertexNormal[1];

//     PRINTF(0)("nr: %i, meshframes: %f, %f, %f, normal: %f, %f\n", i, this->meshes[mesh]->meshFrames[i][0], this->meshes[mesh]->meshFrames[i][1],
//     this->meshes[mesh]->meshFrames[i][2], this->meshes[mesh]->normals[i].vertexNormal[0], this->meshes[mesh]->normals[i].vertexNormal[1]);

      delete vc;
    }
  }

  return this->meshes[mesh]->header->meshFrameNum * this->meshes[mesh]->header->vertexNum * sizeof(MD3VertexCompressed);
}



}







