/*!
 * @file aabb_tree_node.h
 *  Definition of a bounding volume tree
 */

#ifndef _AABB_TREE_NODE_H
#define _AABB_TREE_NODE_H


#include "bv_tree_node.h"
#include "plane.h"

class BoundingVolume;
class AABB;
class OBBTree;
class Plane;
class PNode;


//! A class that represents a bounding volume tree
class AABBTreeNode : public BVTreeNode
{


  public:
    AABBTreeNode(const OBBTree& tree, AABBTreeNode* prev, int depth);
    AABBTreeNode(int depth = 0);
    virtual ~AABBTreeNode();
    void init();

    /**  this function returns the bounding volume of this tree node @return: returns the BV */
    virtual inline const BoundingVolume* getBV() const { return (BoundingVolume*)this->bvElement; }

    void spawnBVTree(Model* model);
    virtual void spawnBVTree(const modelInfo& modelInf, const int* triangleIndexes, int length);

    virtual void collideWith(BVTreeNode* treeNode, WorldEntity* nodeA, WorldEntity* nodeB);
    virtual void drawBV(int depth, int drawMode, const Vector& color = Vector(1,0,0), bool top = true) const;
    void debug() const;

    /**  gets the id of the current child @return id of the child */
    inline const int getID() { return this->nextID++; }
    inline AABB* getAABB() { return this->bvElement; }

  private:
    void calculateBoxAxis(AABB& box, const sVec3D* verticesList, unsigned int length);

    void calculateBoxCovariance(AABB& box, const modelInfo& modelInf, const int* triangleIndexes, int length);
    void calculateBoxEigenvectors(AABB& box, const modelInfo& modelInf, const int* triangleIndexes, int length);
    void calculateBoxAxis(AABB& box, const modelInfo& modelInf, const int* triangleIndexes, int length);
    void forkBox(AABB& box);

    void collideWithOBB(AABBTreeNode* treeNode, WorldEntity* nodeA, WorldEntity* nodeB);
    bool overlapTest(AABB* boxA, AABB* boxB, WorldEntity* nodeA, WorldEntity* nodeB);


  protected:
    AABB*                bvElement;                  //!< the obb element
    AABBTreeNode*        nodePrev;                   //!< ref to the previous (parent) tree node = NULL if first
    AABBTreeNode*        nodeLeft;                   //!< ref to the left tree node
    AABBTreeNode*        nodeRight;                  //!< ref to the right tree node


  private:
    int                 treeIndex;                  //!< Index number of the BV in the tree
    int                 nextID;                     //!< the id of the next child
    int                 depth;                      //!< the depth of the node in the tree
    const OBBTree*      obbTree;                    //!< reference to the obb tree

    const modelInfo*    modelInf;                   //!< pointer to the models modelInfo object
    const int*          triangleIndexes;            //!< indexes to the used model triangles

    Plane               separationPlane;            //!< the separation plane of the obb
    sVec3D              sepPlaneCenter;             //!< only needed to draw plane
    int                 longestAxisIndex;           //!< only needed to draw plane

    /* tmp saving place for obb variables */
    int*                triangleIndexList1;         //!< pointer to the vert data of obbox1
    int*                triangleIndexList2;         //!< pointer to the vert data of obbox1
    int                 triangleIndexLength1;       //!< len vert data obbox1
    int                 triangleIndexLength2;       //!< len vert data obbox2

    WorldEntity*        owner;
};

#endif /* _AABB_TREE_NODE_H */
