/*!
 * @file md3_model.h
 *
 * Code heavely inspired by: JAVA MD3 Model Viewer - A Java based Quake 3 model viewer
 * Copyright (C) 1999 Erwin 'KLR8' Vervaet
 */

#ifndef _MD3_MODEL_H
#define _MD3_MODEL_H

#include <string>
#include "interactive_model.h"


#define FRACTION 0.34f


namespace md3
{


class MD3Data;
class MD3BoneFrame;
class MD3Mesh;
class MD3Tag;
class MD3AnimationCfg;
class MD3Animation;

struct MD3Normal;


typedef enum MD3FrameHandling {
  MD3_ANIM_NEXT = 0,          //!< swith to the next frame
  MD3_ANIM_PREVIOUS,          //!< swith to the previous frame
  MD3_ANIM_REWIND             //!< rewind the animation
};


class MD3Model : public InteractiveModel
{

  public:
    MD3Model(std::string filename, float scaling);
    ~MD3Model();

    virtual void setAnimation(int animNum, int playbackMode = 0) {}
    virtual int getAnimation() { return 0;}

    virtual void tick(float dt);
    virtual void draw() const;


  private:

    void autoAssemblePlayerModel(std::string filename, float scaling);

    void draw(MD3Data* data) const;
    void tick(float dt, MD3Data* data);


    void drawMesh(MD3Mesh* mesh, sVec3D* frame) const;
    void drawVertexNormals(sVec3D* frame, MD3Normal* normals) const;
    void drawBoneFrame(MD3BoneFrame* frame) const;


    MD3BoneFrame* interpolateBoneFrame(MD3Data* data, MD3BoneFrame* currBoneFrame, MD3BoneFrame* nextBoneFrame, float frac);
    sVec3D* interpolateMeshFrame(MD3Data* data, sVec3D* currMeshFrame, sVec3D* nextMeshFrame, float frac, MD3Mesh* mesh, int i);
    MD3Normal* interpolateVertexNormals(MD3Data* data, MD3Normal* curNormals, MD3Normal* nextNormals, float frac, MD3Mesh* mesh, int i);
    float* interpolateTransformation(MD3Data* data, MD3Tag* currFrameTag, MD3Tag* nextFrameTag, float frac, int i);


    void interpolate(MD3Data* data, MD3Animation* anim, int op, bool bInterpolate);

    void visit(MD3Data* data, float time);
    int next(MD3Data* data, int nr);
    int prev(MD3Data* data, int nr);
    void doOp(MD3Data* data, float time);


  private:
    MD3Data*            md3Data;           //!< reference to the md3 model data

    bool                bDrawBones;        //!< draws the bone frames too
    bool                bDrawNormals;      //!< draw the normals

    MD3AnimationCfg*    config;            //!< the config file parsed

    float               time;

};


}

#endif /* _MD3_MODEL_H */
