/*!
 * @file md2Model.h
  *  Definition of an MD2 Model, a model format invented by ID Software.

    We are deeply thankfull for all the wunderfull things id software made for us gamers!

    The md2 file format is structured in a very simple way: it contains animations which are made out of
    frames (so called keyframes). Each frame is a complete draweable model in a specific position.
    A frame is a collection of vertex and its compagnions (normals, texture coordinates).

    A typical model has about 200 frames, the maximum frame count is fixed by MD2_MAX_FRAMES to currently
    512 frames. The maximal vetices count is set to 2048 verteces, not enough?
    You just have to change the MD2_MAX_* values if it doesn't fit your purposes...

    Surface Culling is fully implemented in md2 models: quake2 uses front face culling.
*/

#ifndef _MD3_DATA_H
#define _MD3_DATA_H

#include "base_object.h"

#include "model.h"
#include "material.h"
#include "md_model_structure.h"

#include <map>


//! These are the needed defines for the max values when loading .MD2 files
#define MD3_IDENT                       (('3'<<24) + ('P'<<16) + ('D'<<8) + 'I') //!< the md3 identifier tag in the bin file
#define MD3_VERSION                     15                                       //!< the md2 version in the header



namespace md3
{

  class MD3BoneFrame;
  class MD3Mesh;
  class MD3Normal;
  class MD3Animation;

  //! This holds the header information that is read in at the beginning of the file: id software definition
  typedef struct MD3Header
  {
    int ident;                           //!< This is used to identify the file
    int version;                         //!< The version number of the file (Must be 8)

    char filename[68];                   //!< The filename of the model

    int boneFrameNum;                    //!< number of frames
    int tagNum;                          //!< number of tags
    int meshNum;                         //!< number of mesh
    int maxTexNum;                       //!< number of texture

    int boneFrameStart;                  //!< start of bone frames
    int tagStart;                        //!< start of the tag
    int meshStart;                       //!< mesh start

    int fileSize;                        //!< file size
  };

  //!< holding the informations about the current animation state
  typedef struct MD3AnimationState
  {
    int     currentFrame;          //!< currently rendered animation key frame of this model. Every part has its own current frame!
    int     nextFrame;             //!< next animation key frame of the model.
    float   interpolationFraction; //!< interpolation position between currentFrame and nextFrame [0,1]
  };


  //! class to store the md2 data in
  class MD3Data : public BaseObject
  {
    public:
      MD3Data(const std::string& modelFileName, const std::string& skinFileName, float scale = 1.0f);
      virtual ~MD3Data();

      void addLinkedModel(int tagIndex, MD3Data* child);
      int getTagIndexByName(std::string name);

    private:
      bool loadModel(const std::string& fileName);
      bool loadSkin(const std::string& fileName = "");

      void init();

      int readHeader(FILE* pFile, int fileOffset);
      int readBoneFrames(FILE* pFile, int fileOffset);
      int readTags(FILE* pFile, int fileOffset);
      int readMeshes(FILE* pFile, int fileOffset);


      int readMeshTriangles(FILE* pFile, int fileOffset, int mesh);
      int readMeshTextures(FILE* pFile, int fileOffset, int mesh);
      int readMeshTexVecs(FILE* pFile, int fileOffset, int mesh);
      int readMeshVertices(FILE* pFile, int fileOffset, int mesh);


    public:

      MD3Header*                header;            //!< the header file

      std::map<int, MD3Data*>   sortedMap;         //!< map
      int                       parentTagIndex;    //!< the tag index of the parent model
      MD3Data*                  parent;            //!< the parent model

      std::string               filename;          //!< the name of the file as recorded in the .md3 file

      MD3AnimationState         animationState;    //!< the animation state of this model

      MD3BoneFrame**            boneFrames;        //!< array of bone frames, contains the metadata (bounding box, tags,...) for each frame
      MD3Mesh**                 meshes;            //!< array of meshes in the model. each containing the mesh for each of the animation

      MD3BoneFrame*             tmpBoneFrame;      //!< a temporary bone frame
      sVec3D**                  tmpMesh;           //!< a temporary mesh frame
      MD3Normal**               tmpNormal;         //!< a temporary normals frame
      float**                   tmpMatrix;         //!< a temporary matrix


     /* the animation part */
     MD3Animation*              animation;         //!< animation
     int                        op;                //!< the current operation
     int                        upperBound;        //!< the upper bound
     bool                       bInterpolate;      //!< interpolate the frames
  };


}



#endif /* _MD3_DATA_H */
