/*!
 * @file message_manager.h
 * @brief Definition of MessageManager
*/

#ifndef _MESSAGE_MANAGER_H
#define _MESSAGE_MANAGER_H

#include "synchronizeable.h"

#include <map>
#include <list>

/*
  Protocol:
    int nacks
    int acks[1..nacks]
    int nmsg
    (
      int length
      int number
      int MessageId
      byte * data
    )[1..nmsg]
*/


enum MessageId 
{
  TESTMESSAGEID = 1,
  MSGID_DELETESYNCHRONIZEABLE,
  MSGID_PREFEREDTEAM,
  MSGID_CHANGENICKNAME,
  MSGID_CHATMESSAGE
};

typedef bool (*MessageCallback)( MessageId messageId, byte * data, int dataLength, void * someData, int userId );

enum RecieverType
{
  RT_ALL_NOT_ME = 1,   //!< message is sent to all users
  RT_ALL_ME,           //!< message is sent to all users
  RT_USER,             //!< message is only sent to reciever
  RT_NOT_USER,         //!< message is sent to all but reciever
  RT_SERVER            //!< message is sent to server only
};

//TODO implement priority handling
enum MessagePriority
{
  MP_HIGHBANDWIDTH = 1,  //!< fast and reliable but uses a lot of bandwidth
  MP_LOWBANDWIDTH,       //!< may take a long time to arrive. reliable
  MP_UNRELIABLE          //!< unreliable. low bandwidth 
};

struct NetworkMessage
{
  MessageId        messageId;
  byte *           data;
  int              length;
  int              number;
  MessagePriority  priority;
};

struct MessageUserQueue
{
  std::list<NetworkMessage> messages;
  std::list<int>            toAck;
  std::list<int>            recievedMessages;
};

typedef std::map<int,MessageUserQueue> MessageQueue;

struct MessageHandler
{
  MessageCallback cb;
  MessageId       messageId;
  void *          someData;
};

typedef std::map<MessageId,MessageHandler> MessageHandlerMap;

//! A class for sending messages over network
class MessageManager : public Synchronizeable {
 protected:
   MessageManager();
 public:
   inline static MessageManager * getInstance(){ if (!singletonRef) singletonRef = new MessageManager();  return singletonRef; }
   
   virtual ~MessageManager();
   
   bool registerMessageHandler( MessageId messageId, MessageCallback cb, void * someData );
   
   void sendMessage( MessageId messageId, byte * data, int dataLength, RecieverType recieverType, int reciever, MessagePriority messagePriority );

   virtual int getStateDiff( int userId, byte* data, int maxLength, int stateId, int fromStateId, int priorityTH );
   virtual int setStateDiff( int userId, byte* data, int length, int stateId, int fromStateId );
   virtual void cleanUpUser( int userId );
   virtual void handleSentState( int userId, int stateId, int fromStateId ){}
   virtual void handleRecvState( int userId, int stateId, int fromStateId ){}
   
   void initUser( int userId );


 private:
   static MessageManager *   singletonRef;
   MessageQueue              messageQueue;           //!< stores messages to send
   MessageHandlerMap         messageHandlerMap;      //!< contains handlers for messages

   int                       newNumber;              //!< used to create unique message numbers
   std::list<NetworkMessage> incomingMessageBuffer; 

};

#endif /* _PROTO_CLASS_H */
