/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "terrain_entity.h"
#include "terrain/terrain.h"
#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "spatial_separation.h"

#include "util/loading/resource_manager.h"
#include "model.h"
#include "network_game_manager.h"

#include "material.h"

#include "glincl.h"

#include "state.h"

using namespace std;

CREATE_FACTORY( TerrainEntity, CL_TERRAIN );

/**
 *  standard constructor
 */
TerrainEntity::TerrainEntity (const TiXmlElement* root)
{
	this->init();


	if( root != NULL)
		this->loadParams(root);
	terrain->build( );
}


/**
 *  Constructor for loading a TerrainEntity out of a file
 * @param fileName The file to load data from.

   this either loads out of an OBJ-file, or loads a heightmap if no .obj-extension is found.
*/
TerrainEntity::TerrainEntity(const std::string& fileName )
{
  this->init();

  if (fileName.rfind(".obj" ) != -1 || fileName.rfind(".OBJ") != -1 )
  {
    this->loadModel(fileName);
  }
  else
  {
    // load the hightMap here.
  }
}

/**
 *  a Constructor for the Debug-Worlds
 */
TerrainEntity::TerrainEntity(DebugTerrainEntity debugTerrainEntity)
{
  this->init();
  this->buildDebugTerrainEntity(debugTerrainEntity);
}

/**
 *  standard deconstructor

*/
TerrainEntity::~TerrainEntity ()
{
  if (objectList)
    glDeleteLists(this->objectList, 1);

  if (this->vegetation)
  {
    ResourceManager::getInstance()->unload( this->vegetation );
  }

  if( this->terrain )
    delete terrain;
}


void TerrainEntity::init()
{
  this->setClassID( CL_TERRAIN, "TerrainEntity");
  this->toList(OM_ENVIRON_NOTICK);
  this->toReflectionList();

  this->objectList = 0;
  this->vegetation = NULL;

  this->terrain = new Terrain();

  //this->heightMapMaterial = new Material();
}


void TerrainEntity::loadParams(const TiXmlElement* root)
{
	WorldEntity::loadParams(root);

	LoadParam(root, "scale", this, TerrainEntity, setScale)
		.describe("The scale in x,y,z direction");
	
	LoadParam( root, "lightmap", this, TerrainEntity, loadLightmap )
		.describe("The name of the lightmap.");
		
	LoadParam( root, "elevationmap", this, TerrainEntity, loadElevationmap )
		.describe( "The name of the elevation map. Must be an 8bit image" );
		
}

void TerrainEntity::setScale(float x, float y, float z)
{
 	terrain->setScale( Triple( x, y, z ) );
}

void TerrainEntity::loadElevationmap( const std::string& _eleFile )
{
	terrain->setHeightmap( _eleFile );
}

void TerrainEntity::loadLightmap( const std::string& _lightFile )
{
	Material *mat = new Material( "TERR_LM" );
	mat->setDiffuse( 1.0f, 1.0f, 1.0f );
	mat->setAmbient( 1.0f, 1.0f, 1.0f );
	mat->setDiffuseMap( _lightFile, GL_TEXTURE_2D, 0 );	
	mat->setDiffuseMap( std::string( "maps/gras_128.jpg" ), GL_TEXTURE_2D, 1 );	
	terrain->addMaterial( mat );
}



void TerrainEntity::loadVegetation(const std::string& vegetationFile)
{
  PRINTF(4)("loadVegetation: %s\n", vegetationFile.c_str());
  if (this->vegetation)
    ResourceManager::getInstance()->unload(this->vegetation, RP_LEVEL);
  if (!vegetationFile.empty())
  {
    PRINTF(4)("fetching %s\n", vegetationFile.c_str());
    this->vegetation = dynamic_cast<Model*>(ResourceManager::getInstance()->load(vegetationFile, OBJ, RP_CAMPAIGN));
  }
  else
    this->vegetation = NULL;
}





void TerrainEntity::draw () const
{
	glMatrixMode( GL_MODELVIEW );
  	glPushMatrix();

  /* translate */
   glTranslatef( this->getAbsCoor().x,
                 this->getAbsCoor().y,
                 this->getAbsCoor().z );

	Vector cam = State::getCameraNode()->getAbsCoor();
	
  	if ( this->terrain ) {
		terrain->setCameraPosition( Triple( cam.x, cam.y, cam.z ) ); 
		terrain->draw( );
  	}

  	glPopMatrix();
  /*
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    Vector camera =   State::getCameraNode()->getAbsCoor(); // Go on here ..........!!!

    float height =    heightMap->getHeight(camera.x, camera.z);

    glEnable (GL_COLOR_MATERIAL) ;
    glBegin(GL_QUADS);            // Draw The Cube Using quads
    glColor3f(0.0f,1.0f,0.0f);  // Color Blue
    glVertex3f(camera.x + 63.0f,TerrainEntity->getHeight(camera.x+63.0f, camera.z-10.0f)+13.0f,camera.z-10.0f);      // Top Right Of The Quad (Top)
    glVertex3f(camera.x-63.0f, getHeight(camera.x+63.0f, camera.z-10.0f)+13.0f,camera.z-10.0f);      // Top Left Of The Quad (Top)
    glVertex3f(camera.x-63.0f, getHeight(camera.x+63.0f, camera.z+10.0f)+13.0f, camera.z+10.0f);      // Bottom Left Of The Quad (Top)
    glVertex3f(camera.x+ 63.0f, getHeight(camera.x+63.0f, camera.z+10.0f)+13.0f, camera.z+10.0f);      // Bottom Right Of The Quad (Top)
    glEnd();                      // End Drawing The Plan

    glPopMatrix();*/

}


void TerrainEntity::buildDebugTerrainEntity(DebugTerrainEntity debugTerrainEntity)
{
	terrain->build( );
  /*
  // if the TerrainEntity is the TerrainEntity of Dave
  if (debugTerrainEntity == TERRAINENTITY_DAVE)
  {
    objectList = glGenLists(1);
    glNewList (objectList, GL_COMPILE);

    glColor3f(1.0,0,0);

    int sizeX = 100;
    int sizeZ = 80;
    float length = 1000;
    float width = 200;
    float widthX = float (length /sizeX);
    float widthZ = float (width /sizeZ);

    float height [sizeX][sizeZ];
    Vector normal_vectors[sizeX][sizeZ];


    for ( int i = 0; i<sizeX-1; i+=1)
      for (int j = 0; j<sizeZ-1;j+=1)
        //height[i][j] = rand()/20046 + (j-25)*(j-25)/30;
#ifdef __WIN32__
        height[i][j]=(sin((float)j/3)*rand()*i/182400)*.5;
#else
        height[i][j]=(sin((float)j/3)*rand()*(long)i/6282450500.0)*.5;
#endif

    //Die Huegel ein wenig glaetten
    for (int h=1; h<2;h++)
      for (int i=1;i<sizeX-2 ;i+=1 )
        for(int j=1;j<sizeZ-2;j+=1)
          height[i][j]=(height[i+1][j]+height[i][j+1]+height[i-1][j]+height[i][j-1])/4;

    //Berechnung von normalen Vektoren
    for(int i=1;i<sizeX-2;i+=1)
      for(int j=1;j<sizeZ-2 ;j+=1)
      {
        Vector v1 = Vector (widthX*(1),      height[i][j],      widthZ*(j) );
        Vector v2 = Vector (widthX*(i-1),    height[i-1][j],    widthZ*(j));
        Vector v3 = Vector (widthX*(i),      height[i][j+1],    widthZ*(j+1));
        Vector v4 = Vector (widthX*(i+1),    height[i+1][j],    widthZ*(j));
        Vector v5 = Vector (widthX*(i),      height[i][j-1],    widthZ*(j-1));

        Vector c1 = v2 - v1;
        Vector c2 = v3 - v1;
        Vector c3=  v4 - v1;
        Vector c4 = v5 - v1;
        Vector zero = Vector (0,0,0);
        normal_vectors[i][j]=c1.cross(v3-v5)+c2.cross(v4-v2)+c3.cross(v5-v3)+c4.cross(v2-v4);
        normal_vectors[i][j].normalize();
      }

    glBegin(GL_QUADS);
    int snowheight=3;
    for ( int i = 0; i<sizeX; i+=1)
      for (int j = 0; j<sizeZ;j+=1)
      {
        Vector v1 = Vector (widthX*(i),      height[i][j]-20,       widthZ*(j)  -width/2);
        Vector v2 = Vector (widthX*(i+1),    height[i+1][j]-20,     widthZ*(j)  -width/2);
        Vector v3 = Vector (widthX*(i+1),    height[i+1][j+1]-20,   widthZ*(j+1)-width/2);
        Vector v4 = Vector (widthX*(i),      height[i][j+1]-20,     widthZ*(j+1)-width/2);
        float a[3];
        if(height[i][j]<snowheight)
        {
          a[0]=0;
          a[1]=1.0-height[i][j]/10-.3;
          a[2]=0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
        }
        else
        {
          a[0]=1.0;
          a[1]=1.0;
          a[2]=1.0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);

        }
        glNormal3f(normal_vectors[i][j].x, normal_vectors[i][j].y, normal_vectors[i][j].z);
        glVertex3f(v1.x, v1.y, v1.z);
        if(height[i+1][j]<snowheight)
        {
          a[0]=0;
          a[1] =1.0-height[i+1][j]/10-.3;
          a[2]=0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
        }
        else
        {
          a[0]=1.0;
          a[1]=1.0;
          a[2]=1.0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);

        }
        glNormal3f(normal_vectors[i+1][j].x, normal_vectors[i+1][j].y, normal_vectors[i+1][j].z);
        glVertex3f(v2.x, v2.y, v2.z);
        if(height[i+1][j+1]<snowheight)
        {
          a[0]=0;
          a[1] =1.0-height[i+1][j+1]/10-.3;
          a[2]=0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
        }
        else
        {
          a[0]=1.0;
          a[1]=1.0;
          a[2]=1.0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);


        }
        glNormal3f(normal_vectors[i+1][j+1].x, normal_vectors[i+1][j+1].y, normal_vectors[i+1][j+1].z);
        glVertex3f(v3.x, v3.y, v3.z);
        if(height[i][j+1]<snowheight)
        {
          a[0]=0;
          a[1] =1.0-height[i+1][j+1]/10-.3;
          a[2]=0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
        }
        else
        {
          a[0]=1.0;
          a[1]=1.0;
          a[2]=1.0;
          glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
        }
        glNormal3f(normal_vectors[i][j+1].x, normal_vectors[i][j+1].y, normal_vectors[i][j+1].z);
        glVertex3f(v4.x, v4.y, v4.z);

      }
    glEnd();
    glEndList();
  }
	*/
  if (debugTerrainEntity == TERRAINENTITY_BENSCH)
  {
    /*
      this->model = (OBJModel*) new Model();
    this->model->setName("CUBE");
    this->model->addVertex (-0.5, -0.5, 0.5);
    this->model->addVertex (0.5, -0.5, 0.5);
    this->model->addVertex (-0.5, 0.5, 0.5);
    this->model->addVertex (0.5, 0.5, 0.5);
    this->model->addVertex (-0.5, 0.5, -0.5);
    this->model->addVertex (0.5, 0.5, -0.5);
    this->model->addVertex (-0.5, -0.5, -0.5);
    this->model->addVertex (0.5, -0.5, -0.5);

    this->model->addVertexTexture (0.0, 0.0);
    this->model->addVertexTexture (1.0, 0.0);
    this->model->addVertexTexture (0.0, 1.0);
    this->model->addVertexTexture (1.0, 1.0);
    this->model->addVertexTexture (0.0, 2.0);
    this->model->addVertexTexture (1.0, 2.0);
    this->model->addVertexTexture (0.0, 3.0);
    this->model->addVertexTexture (1.0, 3.0);
    this->model->addVertexTexture (0.0, 4.0);
    this->model->addVertexTexture (1.0, 4.0);
    this->model->addVertexTexture (2.0, 0.0);
    this->model->addVertexTexture (2.0, 1.0);
    this->model->addVertexTexture (-1.0, 0.0);
    this->model->addVertexTexture (-1.0, 1.0);

    this->model->finalize();
    */
  }
}

float TerrainEntity::getHeight( float x, float z )
{
	Triple 	altitude( x-getAbsCoor().x, 0.0f, z-getAbsCoor().z ), 
			normal( 0.0f, 0.0f, 0.0f );
	if ( terrain )
		terrain->getAltitude( altitude, normal );
		
  return altitude.y+getAbsCoor().y;
}
