/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "skybox.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "static_model.h"

#include "material.h"
#include "texture.h"

#include "network_game_manager.h"
#include "converter.h"
#include "util/loading/resource_manager.h"


using namespace std;

CREATE_FACTORY(SkyBox, CL_SKYBOX);

/**
 * Constructs a SkyBox and takes fileName as a map.
 * @param fileName the file to take as input for the SkyBox
*/
SkyBox::SkyBox(const std::string& fileName)
{
  this->preInit();
  if (!fileName.empty())
    this->setTextureAndType(fileName, ".jpg");
  this->postInit();
}

/**
 *  initializes a skybox from a XmlElement
*/
SkyBox::SkyBox(const TiXmlElement* root)
{
  this->preInit();

  if( root != NULL)
    this->loadParams(root);

  this->postInit();
}

void SkyBox::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "Materialset", this, SkyBox, setTexture)
      .describe("Sets the material on the SkyBox. The string must be the path relative to the data-dir, and without a trailing .jpg");

  LoadParam(root, "Size", this, SkyBox, setSize)
      .describe("Sets the Size of the SkyBox (normally this should be 90% of the maximal viewing Distance).");
}

void SkyBox::preInit()
{
  this->setClassID(CL_SKYBOX, "SkyBox");
  this->toList(OM_BACKGROUND);
  this->toReflectionList();
  //this->size = 100.0;
  this->textureSize = 1024.0f;

  for (int i = 0; i < 6; i++)
    {
      this->material[i] = new Material();
      this->material[i]->setIllum(3);
      this->material[i]->setDiffuse(0.0,0.0,0.0);
      this->material[i]->setSpecular(0.0,0.0,0.0);
      this->material[i]->setAmbient(2.0, 2.0, 2.0);

      this->cubeTexture[i] = NULL;
    }
  this->setParentMode(PNODE_MOVEMENT);

  this->textureName = "";
}

void SkyBox::postInit()
{
  this->rebuild();
  
  textureName_handle = registerVarId( new SynchronizeableString( &textureName, &textureName, "textureName") );
  size_handle = registerVarId( new SynchronizeableFloat( &size, &size, "size" ) );
}


/**
 *  default destructor
*/
SkyBox::~SkyBox()
{
  PRINTF(5)("Deleting SkyBox\n");
  for (int i = 0; i < 6; i++)
  {
    if (this->material[i])
      delete this->material[i];
    if (this->cubeTexture[i])
      ResourceManager::getInstance()->unload(this->cubeTexture[i]);
  }
}

void SkyBox::setTexture(const std::string& name)
{
  this->textureName = name;
  this->setTextureAndType (name, "jpg");
};


/**
 * @brief sets A set of textures when just giving a Name and an extension:
 * @param name the prefix of the Name
 * @param extension the file extension (jpg by default)
 * usage: give this function an argument like
 * setTexture("skybox", "jpg");
 * and it will convert this to
 * setTextures("skybox_negx.jpg", "skybox_posx.jpg", "skybox_negy.jpg",
 *             "skybox_posy.jpg", "skybox_negz.jpg", "skybox_posz.jpg");
 */
void SkyBox::setTextureAndType(const std::string& name, const std::string& extension)
{
  std::string negX = name + "_negx." + extension;
  std::string posX = name + "_posx." + extension;

  std::string negY = name + "_negy." + extension;
  std::string posY = name + "_posy." + extension;

  std::string negZ = name + "_negz." + extension;
  std::string posZ = name + "_posz." + extension;

  this->setTextures(negX, posX, negY, posY, negZ, posZ);
}

/**
 * @brief Defines which textures should be loaded onto the SkyBox.
 * @param negX the top texture.
 * @param posX the bottom texture.
 * @param negY the left texture.
 * @param posY the right texture.
 * @param negZ the front texture.
 * @param posZ the back texture.
*/
void SkyBox::setTextures(const std::string& negX, const std::string& posX,
                         const std::string& negY, const std::string& posY,
                         const std::string& negZ, const std::string& posZ)
{
  this->material[0]->setDiffuseMap(negX);
  this->material[1]->setDiffuseMap(posX);
  this->material[2]->setDiffuseMap(negY);
  this->material[3]->setDiffuseMap(posY);
  this->material[4]->setDiffuseMap(negZ);
  this->material[5]->setDiffuseMap(posZ);
  if (GLEW_EXT_texture_cube_map)
    this->loadCubeMapTextures(negX, posX, negY, posY, negZ, posZ);
}

void SkyBox::loadCubeMapTextures(const std::string& posY, const std::string& negY, const std::string& negZ,
                                  const std::string& posZ, const std::string& posX, const std::string& negX)
{
  this->cubeTexture[0] = (Texture*)ResourceManager::getInstance()->load(negX, RP_LEVEL, IMAGE, GL_TEXTURE_CUBE_MAP_NEGATIVE_X_EXT);
  this->cubeTexture[1] = (Texture*)ResourceManager::getInstance()->load(posX, RP_LEVEL, IMAGE, GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT);

  this->cubeTexture[2] = (Texture*)ResourceManager::getInstance()->load(negY, RP_LEVEL, IMAGE, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_EXT);
  this->cubeTexture[3] = (Texture*)ResourceManager::getInstance()->load(posY, RP_LEVEL, IMAGE, GL_TEXTURE_CUBE_MAP_POSITIVE_Y_EXT);

  this->cubeTexture[4] = (Texture*)ResourceManager::getInstance()->load(negZ, RP_LEVEL, IMAGE, GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_EXT);
  this->cubeTexture[5] = (Texture*)ResourceManager::getInstance()->load(posZ, RP_LEVEL, IMAGE, GL_TEXTURE_CUBE_MAP_POSITIVE_Z_EXT);
}

void SkyBox::enableCubeMap()
{
  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP);
  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP);

  glEnable(GL_TEXTURE_CUBE_MAP_EXT);

  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
  glEnable(GL_TEXTURE_GEN_R);

}

void SkyBox::disableCubeMap()
{
  glDisable(GL_TEXTURE_CUBE_MAP);
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_TEXTURE_GEN_S);
  glDisable(GL_TEXTURE_GEN_T);
  glDisable(GL_TEXTURE_GEN_R);

  glDisable(GL_TEXTURE_GEN_S);
  glDisable(GL_TEXTURE_GEN_T);
  glDisable(GL_TEXTURE_GEN_R);
}



/**
 * @param size The new size of the SkyBox

 * do not forget to rebuild the SkyBox after this.
*/
void SkyBox::setSize(float size)
{
  this->size = size;
}



void SkyBox::draw()
{
  glPushAttrib(GL_ENABLE_BIT);
//   glPushAttrib(GL_LIGHTING_BIT);
  glDisable(GL_LIGHTING);
  glDisable( GL_DEPTH_TEST );	
  glDisable(GL_FOG);

  WorldEntity::draw();

  glPopAttrib();

}


/**
 *  rebuilds the SkyBox

   this must be done, when changing the Size of the Skybox (runtime-efficency)
*/
void SkyBox::rebuild()
{
  StaticModel* model = new StaticModel();

  model->addVertex (-0.5*size, -0.5*size, 0.5*size);
  model->addVertex (0.5*size, -0.5*size, 0.5*size);
  model->addVertex (-0.5*size, 0.5*size, 0.5*size);
  model->addVertex (0.5*size, 0.5*size, 0.5*size);
  model->addVertex (-0.5*size, 0.5*size, -0.5*size);
  model->addVertex (0.5*size, 0.5*size, -0.5*size);
  model->addVertex (-0.5*size, -0.5*size, -0.5*size);
  model->addVertex (0.5*size, -0.5*size, -0.5*size);

//   model->addVertexTexture (0.0, 1.0);
//   model->addVertexTexture (1.0, 1.0);
//   model->addVertexTexture (1.0, 0.0);
//   model->addVertexTexture (0.0, 0.0);

  model->addVertexTexture (1.0/this->textureSize, (this->textureSize - 1.0)/this->textureSize);
  model->addVertexTexture ((this->textureSize - 1.0)/this->textureSize, (this->textureSize - 1.0)/this->textureSize);
  model->addVertexTexture ((this->textureSize - 1.0)/this->textureSize, 1.0/this->textureSize);
  model->addVertexTexture (1.0/this->textureSize, 1.0/this->textureSize);


  model->addVertexNormal (0.0, 0.0, 1.0);
  model->addVertexNormal (0.0, 1.0, 0.0);
  model->addVertexNormal (0.0, 0.0, -1.0);
  model->addVertexNormal (0.0, -1.0, 0.0);
  model->addVertexNormal (1.0, 0.0, 0.0);
  model->addVertexNormal (-1.0, 0.0, 0.0);

  model->setMaterial(material[0]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 6,0,4, 0,1,4, 2,2,4, 4,3,4); // back
  model->setMaterial(material[1]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 1,0,5, 7,1,5, 5,2,5, 3,3,5); // front
  model->setMaterial(material[2]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 6,0,1, 7,1,1, 1,2,1, 0,3,1); // bottom
  model->setMaterial(material[3]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 2,0,3, 3,1,3, 5,2,3, 4,3,3); // top
  model->setMaterial(material[4]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 4,2,2, 5,3,2, 7,0,2, 6,1,2); // left
  model->setMaterial(material[5]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 0,0,0, 1,1,0, 3,2,0, 2,3,0); // right

  model->finalize();

  this->setModel(model);
}

void SkyBox::varChangeHandler( std::list< int > & id )
{
  bool somethinChanged = false;
  
  if ( std::find( id.begin(), id.end(), textureName_handle ) != id.end() )
  {
    somethinChanged = true;
    setTexture( textureName );
  }
  
  if ( std::find( id.begin(), id.end(), size_handle ) != id.end() )
  {
    somethinChanged = true;
  }
  
  rebuild();
  
  WorldEntity::varChangeHandler( id );
}
