/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GUI

#include "glgui_table.h"
#include "debug.h"

namespace OrxGui
{
  /**
   * @brief standard constructor
   */
  GLGuiTable::GLGuiTable (unsigned int rows, unsigned int columns)
  {
    this->init();
    this->setRowCount(rows);
    this->setColumnCount(columns);
  }


  /**
   * @brief standard deconstructor
   */
  GLGuiTable::~GLGuiTable()
  {}

  /**
   * @brief initializes the GUI-element
   */
  void GLGuiTable::init()
  {
    this->setClassID(CL_GLGUI_TABLE, "GLGuiTable");

    this->setFocusable(true);
    this->setClickable(true);

    this->resize();
  }


  void GLGuiTable::setRowCount(unsigned int rowCount)
  {

    unsigned int currentRowCount = this->rowCount();
    this->_entries.resize(rowCount);
    for (unsigned int i = currentRowCount; i < this->rowCount(); ++i)
    {
      this->_entries[i].resize(columnCount(), LimitedWidthText("fonts/final_frontier.ttf", 20));
      for (unsigned int j = 0; j < columnCount(); ++j)
        this->applyTextSettings(i, j, &this->_entries[i][j]);
    }

    assert(this->checkIntegrity());
    this->debug();

    if (!this->isVisible())
      this->hiding();
  }

  void GLGuiTable::setColumnCount(unsigned int columnCount)
  {
    unsigned int i;
    unsigned int currentColumnCount = this->columnCount();

    // setup Headers.
    this->_headers.resize(columnCount, LimitedWidthText("fonts/final_frontier.ttf", 20));
    for (i = currentColumnCount; i < columnCount; ++i)
      this->applyTextSettings(0, i, &this->_headers[i]);

    for (i = 0; i < rowCount(); i++)
    {
      this->_entries[i].resize(columnCount, LimitedWidthText("fonts/final_frontier.ttf", 20));
      for (unsigned int j = currentColumnCount; j < columnCount; ++j)
        this->applyTextSettings(i, j, &this->_entries[i][j]);

    }
    assert(this->checkIntegrity());
    this->debug();

    if (!this->isVisible())
      this->hiding();
  }




  void GLGuiTable::setHeader(unsigned int number, const std::string& headerName)
  {
    if (number >= this->columnCount())
      this->setColumnCount(number + 1);
    this->_headers[number].setText(headerName);
  }

  void GLGuiTable::setHeader(const std::vector<std::string>& headerNames)
  {
    for (unsigned int i = 0; i < headerNames.size(); ++i)
      this->setHeader(i, headerNames[i]);
  }

  void GLGuiTable::setEntry(unsigned int column, unsigned int row, const std::string& name)
  {
    if (column >= this->columnCount() )
      this->setColumnCount(column + 1);
    if (row >= this->rowCount() )
      this->setRowCount(row + 1);
    this->_entries[row][column].setText(name);
  }

  void GLGuiTable::setColumnWidth(unsigned int column, float size)
  {
    this->_headers[column].setLineWidth(size);
    for (unsigned int i = 0; i < this->rowCount(); ++i)
      this->_entries[i][column].setLineWidth(size);

  }

  /// TODO.
  void GLGuiTable::setRowHeight(unsigned int row, unsigned int size)
{}




  /**
   * removes the focus from this Widget.
   */
  void GLGuiTable::removedFocus()
  {
    GLGuiWidget::removedFocus();
  }



  /**
   * @brief Processes an Event.
   * @param event The event to be processed
   * @return true if the event was catched.
   */
  bool GLGuiTable::processEvent(const Event& event)
  {
    return false;
  }


  /**
   * @brief Resizes the Widget to the new Size-constraints.
   */
  void GLGuiTable::resize()
  {
    GLGuiWidget::resize();
  }

  void GLGuiTable::updateFrontColor()
  {
    for (unsigned int i = 0; i < this->columnCount(); ++i)
    {
      this->_headers[i].setColor(this->foregroundColor());
      for (unsigned int j = 0; j < this->rowCount(); ++j)
        this->_entries[j][i].setColor(this->foregroundColor());
    }
  }

  void GLGuiTable::hiding()
  {
    for (unsigned int i = 0; i < this->columnCount(); ++i)
    {
      this->_headers[i].setVisibility(false);
      for (unsigned int j = 0; j < this->rowCount(); ++j)
        this->_entries[j][i].setVisibility(false);
    }
  }

  void GLGuiTable::showing()
  {
    for (unsigned int i = 0; i < this->columnCount(); ++i)
    {
      this->_headers[i].setVisibility(true);
      for (unsigned int j = 0; j < this->rowCount(); ++j)
        this->_entries[j][i].setVisibility(true);
    }
  }


  /**
   * @brief ticks the Table
   * @param dt the time passed.
   */
  void GLGuiTable::tick(float dt)
{}

  /**
   * @brief draws the GLGuiTable
   */
  void GLGuiTable::draw() const
  {
    this->beginDraw();
    GLGuiWidget::draw();

    //     this->frontMaterial().select();
    //     GLGuiWidget::drawRect(this->frontRect());

    this->endDraw();
  }


  /**
   * @brief applies the GLGuiNotifiers Settings to a single Text of the GLGuiNotifier.
   * @param text the Text to apply the settings to.
   */
  void GLGuiTable::applyTextSettings(unsigned int row, unsigned int column, LimitedWidthText* text)
  {
    text->setSize(this->textSize());
    //text->setLineWidth(  );
    text->setFont("fonts/final_frontier.ttf", (int)this->textSize());

    text->setColor(this->foregroundColor());
    if (text->getParent2D() != this)
      text->setParent2D(this);
  }


  void GLGuiTable::debug() const
  {
    PRINT(0)("Table: Size = %dx%d\n", this->rowCount(), this->columnCount());

    for (unsigned int i = 0; i < this->rowCount(); ++i)
      PRINT(0)("line %d: columns %d\n", i, this->_entries[i].size());
  }


  bool GLGuiTable::checkIntegrity() const
  {
    bool retVal = true;
    if (rowCount() != this->_entries.size())
    {
      PRINTF(1)("ROW COUNT WRONG (is %d should be %d)\n", rowCount(), this->_entries.size());
      retVal = false;
    }
    if (columnCount() != this->_headers.size())
    {
      PRINTF(1)("COLUMN COUNT WRONG (is %d should be %d)\n", columnCount(), this->_headers.size());
      retVal = false;
    }
    for (unsigned int i = 0; i < this->rowCount(); ++i)
      if (this->_entries[i].size() != this->columnCount())
    {
      PRINTF(1)("COLUMN-COUNT OF ROW %d WRONG (is %d should be %d)\n", i, this->_entries[i].size(), this->columnCount());
      retVal = false;
    }
    return retVal;
  }

}
