/*
	orxonox - the future of 3D-vertical-scrollers
 
	Copyright (C) 2006 orx
 
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.
 
	### File Specific:
	main programmer: Marco Biasini

 */

#ifndef _TERRAIN_H
#define _TERRAIN_H

#define USE_VBO
#ifdef USE_VBO
#include "buffer_broker.h"
#endif
#include "types.h"
#include "terrain_page.h"
#include "terrain_quad.h"
#include <cstdlib>
#include <string>
#include <list>
#include "material.h"
#include "texture.h"
#include "frustum.h"


typedef struct {
	int 		width;
	int 		height;
	int			pitch;
	UByte		*data;
} Heightfield, *pHeightfield;

class Terrain;

typedef Terrain *pTerrain;

class TerrainPage;

class Terrain {
	public:
	
		/**
		 * The amount of geometry rendered is largely depending on this constant. So chose a
		 * reasonable value...
		 */
		inline float getDetail() { return 0.05f; }

	
		inline bool debug()
		{
			return false;
		}
		
		inline pTerrainPage getActiveList() { return activePages; }

	
		inline void setCameraPosition( const Triple& _cameraPosition ) 
		{ 
			cameraPosition = _cameraPosition; 
		}
		
		inline const Triple& getCameraPosition() const
		{
			return cameraPosition;
		}
	
		void getAltitude( Triple& _alt, Triple& _normal );
		
		void showPages( int _x0, int _z0, int _width, int _height );
					
		/**
		 * Create a new terrain.
		 */
		Terrain( )
		{ 
			pageSize = 17;
			scale  = Triple( 1.0f, 3.0f, 1.0f );
			frustum = new Frustum();
			activatedCount = deactivatedCount = 0;
#ifdef USE_VBO
			broker = new BufferBroker( 400, pageSize*pageSize*sizeof( Vertex ), 
				pageSize*pageSize*sizeof( short )*3 );
#endif			
		}
		
		/**
		 * Draws the terrain.
		 */
		void draw( );

		void setPageSize( const Triple& _page ) { }
		
		void build();
		
		inline void setActiveList( pTerrainPage _page )
		{
			activePages = _page;
		}
		inline pFrustum getViewingFrustum()
		{
			return frustum;
		}
		
		/**
		 * Returns the normalized ( range from [0..1] ) altitude from the heightmap for the given 
		 * coordinates.
		 */
		inline void incActivated() { activatedCount++; }
		inline void incDeactivated() { deactivatedCount++; }
		inline float getAltitude( int _x, int _z ) const;
		
		inline void addMaterial( Material *_material )
		{
			materials.push_back( _material );
		}
		
		/*inline void setLightmap( const std::string &_lightmap )
		{
			lightmapSource = _lightmap;
		}*/
		inline void setHeightmap( const std::string &_heightmap ) 
		{ 
			heightmapSource = _heightmap; 
		}
		
		inline void setPageSize( int _pageSize )
		{
			pageSize = _pageSize;
		}
		
		inline const Triple getScale() const { return scale; }
		
		inline int getPageSize() const 
		{
			return pageSize;
		}
		
		inline void getCoord( int _x, int _z, TexCoord& _coord) const
		{
			_coord.u = ( ( float )_x )/( heightfield.width-1 );
			_coord.v = ( ( float )_z )/( heightfield.height-1 );			
		}
		
		inline void setScale( const Triple& _scale )
		{
			scale = _scale;
		}
#ifdef USE_VBO
		inline pBufferBroker getBufferBroker() { return broker; }
#endif		
		inline TerrainPage *getPage( int _x, int _z )
		{
			return pages[pagesX*_z+_x];
		}
	protected:
			
		/**
		 * convenience method to create a new terrain page at position offset
		 * p = ( _xOffset, _zOffset ). 
		 */
		pTerrainPage createPage( int _xOffset, int _zOffset ) const;				
		
		/**
		 * Creates the quad tree structure for fast culling of the terrain pages.
		 */		
		pTerrainQuad createQuadTree( int _x0, int _z0, int _x1, int _z1, int _depth = 0 );
		
		/**
		 * Walks the quad tree to determine which pages are visible, e.g. are in the viewing
		 * frustum of the camera.
		 */
		void determineVisiblePages( pTerrainQuad _node );						
		
#ifdef USE_VBO
		pBufferBroker			broker;
#endif
		pTerrainQuad			root;		// The quad-tree root node.
		pTerrainPage			*pages;		// the references to all pages
		std::string				heightmapSource;
		std::string				lightmapSource;
		Heightfield				heightfield;
		Triple					scale;
		int 					pagesX, 
								pagesZ;
		int						pageSize;	
		int						cullCount;
		
		int						activatedCount, 
								deactivatedCount; // For debugging and statistics
		Texture					*tex;
		Triple					cameraPosition;
		pFrustum				frustum;
		pTerrainPage			activePages;
		std::vector<Material*>	materials;
};

inline float Terrain::getAltitude( int _x, int _z ) const
{
	return heightfield.data[heightfield.pitch*_z+_x]/255.0f;	
}

#endif
