/*!
 * @file glgui_style.h
 * @brief Definition of the OpenGL-GUI Style of a Class.
*/

#ifndef _GLGUI_STYLE_H
#define _GLGUI_STYLE_H

#include "base_object.h"

#include "glgui_defs.h"

#include "material.h"
#include "font.h"
#include "texture.h"
#include "color.h"

namespace OrxGui
{
  //! A class for Defining Styles to the opengl-gui.
  class GLGuiStyle : public BaseObject
  {
  public:
    GLGuiStyle(const TiXmlElement* root = NULL);
    virtual ~GLGuiStyle();

    ////////////////////////////////
    /// Retrieve Current Values. ///
    ////////////////////////////////
    /** @returns current left borderWidth */
    inline float borderLeft() const { return _currentStyle._borderLeft; }
    /** @returns current right borderWidth */
    inline float borderRight() const { return _currentStyle._borderRight; }
    /** @returns current top borderWidth */
    inline float borderTop() const { return _currentStyle._borderTop; }
    /** @returns burrent bottom borderWidth */
    inline float borderBottom() const { return _currentStyle._borderBottom; }


    /** @returns current textSize */
    inline float textSize() const { return _currentStyle._textSize; }
    /** @returns the Background Color */
    inline const Color& backgroundColor() const { return _currentStyle._background.diffuseColor(); }
    /** @returns the current Background Material. */
    inline const Material& background() const { return _currentStyle._background; }
    /** @returns the current background Texture. */
    inline const Texture& backgroundTexture() const { return _currentStyle._background.diffuseTexture(); }
    /** @returns the current foreground Color */
    inline const Color& foregroundColor() const { return _currentStyle._foreground.diffuseColor(); }
    /** @returns the current ForeGroung Material. */
    inline const Material& foreground() const { return _currentStyle._foreground; }


    /** @returns FeaturePosition */
    inline FeaturePosition featurePosition() const { return _featurePosition; }
    /** @returns the font */
    inline const Font* const font() const { return _font; }
    /** @returns true if the Element is Animated */
    inline bool animated() const { return _animated; }
    /** @returns true if State-Changes are animated */
    inline bool animatedStateChanges() const { return _animatedStateChanges; }


    ///////////////////////////////////////////////////////////////
    /// Retrieve Values for the Saved Values inside the States. ///
    ///////////////////////////////////////////////////////////////
    /** @returns left borderWidth @param state the State to retrieve from */
    inline float borderLeft(OrxGui::State state) const { return _style[state]._borderLeft; }
    /** @returns right borderWidth @param state the State to retrieve from */
    inline float borderRight(OrxGui::State state) const { return _style[state]._borderRight; }
    /** @returns top borderWidth @param state the State to retrieve from */
    inline float borderTop(OrxGui::State state) const { return _style[state]._borderTop; }
    /** @returns bottom borderWidth @param state the State to retrieve from */
    inline float borderBottom(OrxGui::State state) const { return _style[state]._borderBottom; }

    /** @returns textSize @param state the State to retrieve from */
    inline float textSize(OrxGui::State state) const { return _style[state]._textSize; }
    /** @returns the Background Color @param state the State to retrieve from */
    inline const Color& backgroundColor(OrxGui::State state) const { return _style[state]._background.diffuseColor(); }
    /** @returns the Background Material. @param state the state to retrieve from */
    inline const Material& background(OrxGui::State state) const { return _style[state]._background; }
    /** @returns background Texture. @param state the State to retrieve from */
    inline const Texture& backgroundTexture(OrxGui::State state) const { return _style[state]._background.diffuseTexture(); }
    /** @returns the foreground Color @param state the State to retrieve from */
    inline const Color& foregroundColor(OrxGui::State state) const { return _style[state]._foreground.diffuseColor(); }
    /** @returns the ForeGroung Material. @param state the state to retrieve from */
    inline const Material& foreground(OrxGui::State state) const { return _style[state]._foreground; }


    /// SETUP
    void reset();
    void loadParams(const TiXmlElement* root);

    void setBorderLeft(float value);
    void setBorderLeft(float value, OrxGui::State state);
    void setBorderLeftS(float value, const std::string& stateName);

    void setBorderRight(float value);
    void setBorderRight(float value, OrxGui::State state);
    void setBorderRightS(float value, const std::string& stateName);

    void setBorderTop(float value);
    void setBorderTop(float value, OrxGui::State state);
    void setBorderTopS(float value, const std::string& stateName);

    void setBorderBottom(float value);
    void setBorderBottom(float value, OrxGui::State state);
    void setBorderBottomS(float value, const std::string& stateName);

    void setTextSize(float value);
    void setTextSize(float value, OrxGui::State state);
    void setTextSizeS(float value, const std::string& stateName);

    void setBackgroundColor(const Color& color);
    void setBackgroundColor(const Color& color, OrxGui::State state);
    void setBackgroundColorS(float r, float g, float b, float a, const std::string& stateName);

    void setBackgroundTexture(const Texture& texture);
    void setBackgroundTexture(const std::string& textureName);
    void setBackgroundTexture(const Texture& texture, OrxGui::State state);
    void setBackgroundTexture(const std::string& textureName, const std::string& stateName);

    void setForegroundColor(const Color& color);
    void setForegroundColor(const Color& color, OrxGui::State state);
    void setForegroundColorS(float r, float g, float b, float a, const std::string& stateName);

    inline Color& foregroundColor(OrxGui::State state= GLGUI_DEFAULT_STYLE) { return _style[state]._foreground.diffuseColor(); }
    inline Color& backgroundColor(OrxGui::State state= GLGUI_DEFAULT_STYLE) { return _style[state]._background.diffuseColor(); }
    inline Material& background(OrxGui::State state = GLGUI_DEFAULT_STYLE) { return _style[state]._background; }
    inline Material& foreground(OrxGui::State state = GLGUI_DEFAULT_STYLE) { return _style[state]._foreground; }

    void loadBackgroundMaterial(const Material& material);
    void loadBackgroundMaterial(const Material& material, OrxGui::State state);
    void loadBackgroundMaterial(const TiXmlElement* element);
    void loadBackgroundMaterial(const TiXmlElement* element, OrxGui::State state);
    void loadBackgroundMaterialS(const TiXmlElement* element, const std::string& stateName);

    void loadForegroundMaterial(const Material& material);
    void loadForegroundMaterial(const Material& material, OrxGui::State state);
    void loadForegroundMaterial(const TiXmlElement* element, OrxGui::State state);
    void loadForegroundMaterialS(const TiXmlElement* element, const std::string& stateName);

    void setFeaturePosition(FeaturePosition featurePosition);
    void setFeaturePositionS(const std::string& featurePosition);

    void setFont(Font* font);
    void setFont(const std::string& fontName);

    void setAnimated(bool animated);
    void setAnimatedStateChanges(bool animated);


    void tick(float dt);
    void switchState(OrxGui::State state);


  private:
    bool getState(const std::string& stateName, OrxGui::State* state);

  private:
    typedef struct
    {
      float             _borderLeft;           //!< The Distance to the left Border of the widget, before any internal Element starts.
      float             _borderRight;          //!< The Distance to the right Border of the widget, before any internal Element starts.
      float             _borderTop;            //!< The Distance to the top Border of the widget, before any internal Element starts
      float             _borderBottom;         //!< The Distance to the bottom Border of the widget, before any internal Element starts

      float             _textSize;             //!< The TextSize of the Widget.

      Material          _background;           //!< The Background Material of the Widget.

      Material          _foreground;           //!< The foreground Material of the Widget.
    }
    StatedStyle;


    StatedStyle         _style[GLGUI_STATE_COUNT]; //!< Styles configured for different States

    FeaturePosition     _featurePosition;      //!< The Position a Feature will be layed at (checkbox(box), slider(text),...)
    Font*               _font;                 //!< The Font used in the current Widget.


    /// ANIMATION STUFF:
    bool                _animated;             //!< If the Widget is animated (Texture might be an AnimatedTexture.)
    bool                _animatedStateChanges; //!< If the Transitions between States are Animated automatically.

    float               _animationCycle;
    float               _animationDuration;
    StatedStyle         _currentStyle;
    OrxGui::State       _currentState;
  };
}
#endif /* _GLGUI_STYLE_H */
