/*!
 * @file file.h
 * @brief Definition of File Handler class
 */

#ifndef __FILE_H_
#define __FILE_H_


#include <string>
struct stat;

//! A Class to Handle Files.
class File
{
  public:
  //! How the File should be opened.
  typedef enum
  {
    ReadOnly,      //!< ReadOnly mode
    WriteOnly,     //!< WriteOnly mode
    ReadWrite,     //!< Read and Write mode together
    Append,        //!< Append at the end.
  } OpenMode;

public:
  File();
  File(const std::string& fileName);
  File(const File& file);
  virtual ~File();

  /// Set-Up
  void setFileName(const std::string& fileName);
  File& operator=(const std::string& fileName);
  File& operator=(const File& file);

  /// Comparison
  bool operator==(const std::string& fileName) const;
  bool operator==(const File& file) const;

  virtual bool open(OpenMode mode);
  virtual bool close();
  int handle() const { return this->_handle; };

  /** @returns the FileName of this File */
  const std::string& name() const { return this->_name; };

  /// Testing
  bool exists() const;
  bool isLink() const;
  bool isFile() const;
  bool isDirectory() const;
  bool isReadable() const;
  bool isWriteable() const;
  bool isExecutable() const;


  /// Operate on the FileSystem
  bool copy(const File& destination);
  bool rename(const File& destination);
  bool touch();
  bool remove();

  /// Transformations
  static void relToAbs(std::string& relFileName);
  static void absToRel(std::string& absFileName);
  static const std::string& cwd();

  private:
    void init();
    void statFile();
    void homeDirCheck(std::string& fileName);

  private:
    int                 _handle;          //!< The FileHandle (if set).
    std::string         _name;            //!< The Name of the File.
    stat*               _status;          //!< The Stat of the File.

    static std::string  _cwd;             //!< The currend Working directory.

};

#endif /* __FILE_H_ */
