/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: ...
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "glgui_style.h"

#include "loading/load_param.h"

namespace OrxGui
{


  /**
   * @brief standard constructor
   * @param root the XML-Element to load settings from
   */
  GLGuiStyle::GLGuiStyle (const TiXmlElement* root)
  {
    this->_font = NULL;
    this->_currentState = OrxGui::Normal;
    this->_animationCycle = -1.0;
    this->_animationDuration = 1.0;

    /// current-style
    this->_currentStyle._borderLeft = 1.0;
    this->_currentStyle._borderRight = 1.0;
    this->_currentStyle._borderTop = 1.0;
    this->_currentStyle._borderBottom = 1.0;
    this->_currentStyle._textSize = 20.0;

    this->reset();


    if (root != NULL)
      this->loadParams(root);

  }


  /**
   * @brief standard deconstructor
   */
  GLGuiStyle::~GLGuiStyle ()
  {
    // delete what has to be deleted here
  }

  /**
   * @brief resets the Style to the default Settings.
   */
  void GLGuiStyle::reset()
  {
    this->setBorderLeft(1.0);
    this->setBorderRight(1.0);
    this->setBorderTop(1.0);
    this->setBorderBottom(1.0);

    this->setTextSize(20.0);
    this->setBackgroundColor(1.0);
    this->setForegroundColor(1.0);

    this->setFeaturePosition(FeatureLeft);
    this->setFont(NULL);

    this->setAnimated(true);
    this->setAnimatedStateChanges(true);
  }

  /**
   * @brief loads Parameters for a Style from XML
   * @param root the XML-Element to load from.
   */
  void GLGuiStyle::loadParams(const TiXmlElement* root)
  {
    LoadParam(root, "border-left", this, GLGuiStyle, setBorderLeft);
    LoadParam(root, "border-right", this, GLGuiStyle, setBorderRight);
    LoadParam(root, "border-top", this, GLGuiStyle, setBorderTop);
    LoadParam(root, "border-bottom", this, GLGuiStyle, setBorderBottom);

    LoadParam(root, "text-size", this, GLGuiStyle, setTextSize);
    LoadParam(root, "background-color", this, GLGuiStyle, setBackgroundColorS);
    LoadParam(root, "foreground-color", this, GLGuiStyle, setForegroundColorS);

//    LoadParamXML(root, "backmat", this, GLGuiStyle, loadBackgroundMaterial);
//    LoadParamXML(root, "frontmat", this, GLGuiStyle, loadForegroundMaterial);

    LoadParam(root, "feature-position", this, GLGuiStyle, setFeaturePositionS);
    LoadParam(root, "Font", this, GLGuiStyle, setFont);

    LoadParam(root, "animated", this, GLGuiStyle, setAnimated);
    LoadParam(root, "animated-state-changes", this, GLGuiStyle, setAnimatedStateChanges);
  }

  /**
   * @brief sets the Width of the left border for all States
   * @param value the borderWidth
   */
  void GLGuiStyle::setBorderLeft(float value)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setBorderLeft(value, (OrxGui::State)i);
  }

  /**
   * @brief sets the Width of the left border.
   * @param value the borderWidth
   * @param state the State to set the borderwidth to
   */
  void GLGuiStyle::setBorderLeft(float value, OrxGui::State state)
  {
    _style[state]._borderLeft = value;
  }

  /**
   * @brief sets the Width of the left border.
   * @param value the borderWidth
   * @param stateName the State to set the borderwidth to
   */
  void GLGuiStyle::setBorderLeftS(float value, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setBorderLeft(value, state);
    else
      this->setBorderLeft(value);
  }

  /**
   * @brief sets the Width of the right border for all states.
   * @param value the borderWidth
   */
  void GLGuiStyle::setBorderRight(float value)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setBorderRight(value, (OrxGui::State)i);
  }

  /**
   * @brief sets the Width of the right border.
   * @param value the borderWidth
   * @param state the State to setup.
   */
  void GLGuiStyle::setBorderRight(float value, OrxGui::State state)
  {
    _style[state]._borderRight = value;
  }

  /**
   * @brief sets the Width of the right border.
   * @param value the borderWidth
   * @param stateName the State to setup.
   */
  void GLGuiStyle::setBorderRightS(float value, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setBorderRight(value, state);
    else
      this->setBorderRight(value);
  }


  /**
   * @brief sets the Width of the top border for all states.
   * @param value the borderWidth
   */
  void GLGuiStyle::setBorderTop(float value)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setBorderTop(value, (OrxGui::State)i);
  }

  /**
   * @brief sets the Width of the top border.
   * @param value the borderWidth
   * @param state the State to setup.
   */
  void GLGuiStyle::setBorderTop(float value, OrxGui::State state)
  {
    _style[state]._borderTop = value;
  }

  /**
   * @brief sets the Width of the top border.
   * @param value the borderWidth
   * @param stateName the State to setup.
   */
  void GLGuiStyle::setBorderTopS(float value, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setBorderTop(value, state);
    else
      this->setBorderTop(value);
  }


  /**
   * @brief sets the Width of the bottom border for all states.
   * @param value the borderWidth
   */
  void GLGuiStyle::setBorderBottom(float value)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setBorderBottom(value, (OrxGui::State)i);
  }

  /**
   * @brief sets the Width of the bottom border.
   * @param value the borderWidth
   * @param state the State to setup.
   */
  void GLGuiStyle::setBorderBottom(float value, OrxGui::State state)
  {
    _style[state]._borderBottom = value;
  }

  /**
   * @brief sets the Width of the bottom border for all states.
   * @param value the borderWidth
   * @param stateName the State to setup.
   */
  void GLGuiStyle::setBorderBottomS(float value, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setBorderBottom(value, state);
    else
      this->setBorderBottom(value);
  }


  /**
   * @brief sets the TextSize for all states.
   * @param value the TextSize
   */
  void GLGuiStyle::setTextSize(float value)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setTextSize(value, (OrxGui::State)i);
  }

  /**
   * @brief sets the TextSize.
   * @param value the TextSize.
   * @param state: the State to setup
   */
  void GLGuiStyle::setTextSize(float value, OrxGui::State state)
  {
    _style[state]._textSize = value;
  }

  /**
   * @brief sets the TextSize.
   * @param value the TextSize.
   * @param stateName: the State to setup
   */
  void GLGuiStyle::setTextSizeS(float value, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setTextSize(value, state);
    else
      this->setTextSize(value);
  }


  /**
   * @brief sets the Background Color for all States.
   * @param color the Color.
   */
  void GLGuiStyle::setBackgroundColor(const Color& color)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setBackgroundColor(color, (OrxGui::State)i);
  }

  /**
   * @brief sets the Background Color.
   * @param color the Color.
   * @param state: the State to setup
   */
  void GLGuiStyle::setBackgroundColor(const Color& color, OrxGui::State state)
  {
    _style[state]._background.setDiffuseColor(color);
  }

  /**
   * @brief sets the Background Color.
   * @param r the Color's red part.
   * @param g the Color's green part.
   * @param b the Color's blue part.
   * @param a the Color's alpha part.
   * @param stateName: the State to setup
   */
  void GLGuiStyle::setBackgroundColorS(float r, float g, float b, float a, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setBackgroundColor(Color(r,g,b,a), state);
    else
      this->setBackgroundColor(Color(r,g,b,a));
  }


  /**
   * @brief sets the Background Texture for all States.
   * @param texture the Texture.
   */
  void GLGuiStyle::setBackgroundTexture(const Texture& texture)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setBackgroundTexture(texture, (OrxGui::State)i);
  }

    /**
     * @brief sets the Background Texture to all States.
     * @param textureName the Texture's fileName.
     */
  void GLGuiStyle::setBackgroundTexture(const std::string& textureName)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      _style[i]._background.setDiffuseMap(textureName);
  }

  /**
   * @brief sets the Background Texture.
   * @param texture the Texture.
   * @param state the State to setup.
   */
  void GLGuiStyle::setBackgroundTexture(const Texture& texture, OrxGui::State state)
  {
    _style[state]._background.setDiffuseMap(texture);
  }



  /**
   * @brief sets the Background Texture.
   * @param texture the Texture.
   * @param stateName the State to setup.
   */
  void GLGuiStyle::setBackgroundTexture(const std::string& textureName, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      ; /// FIXME this->setBackgroundTexture(textureName, state);
    else
      ; ///    this->setBackgroundTexture(textureName);
  }


  /**
   * @brief sets the Foreground Color for all States.
   * @param color the Color.
   */
  void GLGuiStyle::setForegroundColor(const Color& color)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      setForegroundColor(color, (OrxGui::State)i);
  }

  /**
   * @brief sets the Foreground Color.
   * @param color the Color.
   * @param state the State to setup
   */
  void GLGuiStyle::setForegroundColor(const Color& color, OrxGui::State state)
  {
    _style[state]._foreground.setDiffuseColor(color);
  }

  /**
   * @brief sets the Foreground Color.
   * @param r the Color's red part.
   * @param g the Color's green part.
   * @param b the Color's blue part.
   * @param a the Color's alpha part.
   * @param stateName: the State to setup
   */
  void GLGuiStyle::setForegroundColorS(float r, float g, float b, float a, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->setForegroundColor(Color(r,g,b,a), state);
    else
      this->setForegroundColor(Color(r,g,b,a));
  }


  void GLGuiStyle::loadBackgroundMaterial(const Material& material)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      this->loadForegroundMaterial(material, (OrxGui::State)i);
  }

  void GLGuiStyle::loadBackgroundMaterial(const Material& material, OrxGui::State state)
  {
    this->_style[state]._background = material;
  }

  void GLGuiStyle::loadBackgroundMaterial(const TiXmlElement* element)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      this->_style[i]._background.loadParams(element);
  }

  void GLGuiStyle::loadBackgroundMaterial(const TiXmlElement* element, OrxGui::State state)
  {
    this->_style[state]._background.loadParams(element);
  }
  void GLGuiStyle::loadBackgroundMaterialS(const TiXmlElement* element, const std::string& stateName)
  {
    OrxGui::State state;
    if (getState(stateName, &state))
      this->loadBackgroundMaterial(element, state);
    else
      this->loadBackgroundMaterial(element);
  }

  void GLGuiStyle::loadForegroundMaterial(const Material& material)
  {}
  void GLGuiStyle::loadForegroundMaterial(const Material& material, OrxGui::State state)
  {}
  void GLGuiStyle::loadForegroundMaterial(const TiXmlElement* element, OrxGui::State state)
  {}
  void GLGuiStyle::loadForegroundMaterialS(const TiXmlElement* element, const std::string& stateName)
  {}


  /**
   * @brief sets the Feature-Position.
   * @param featurePosition the Feature-Position.
   */
  void GLGuiStyle::setFeaturePosition(FeaturePosition featurePosition)
  {
    this->_featurePosition = featurePosition;
  }

  /**
   * @brief sets the Feature-Position by converting from a String.
   * @param featurePosition the Feature-Position.
   */
  void GLGuiStyle::setFeaturePositionS(const std::string& featurePosition)
  {
    for (unsigned int i = 0; i < 4; ++i)
    {
      if (featurePosition == FeaturePositionString[i])
      {
        this->setFeaturePosition((FeaturePosition)i);
      }
    }
  }

  /**
   * @brief sets the Font.
   * @param font the Font.
   */
  void GLGuiStyle::setFont(Font* font)
  {
    this->_font = font;
  }

  /**
   * @brief sets the font from a Font-Name.
   * @param fontName the FileName of the Font.
   */
  void GLGuiStyle::setFont(const std::string& fontName)
  {
    //this->font = new Font(fontName);
  }

  /**
   * @brief if Animation should be turned on.
   * @param animated true if on, false otherwise.
   */
  void GLGuiStyle::setAnimated(bool animated)
  {
    this->_animated = animated;
  }

  /**
   * @brief sets the AnimatedState.
   * @param animated: it states-changes should animate true, otherwise false.
   */
  void GLGuiStyle::setAnimatedStateChanges(bool animated)
  {
    this->_animatedStateChanges = animated;
  }


  void GLGuiStyle::tick(float dt)
  {

  }

  void GLGuiStyle::switchState(OrxGui::State state)
  {
    this->_currentStyle = this->_style[state];
    this->_currentState = state;
    printf("Switching to state %s\n", OrxGui::StateString[state].c_str());
  }


  /**
   * @param stateName the Name of a State.
   * @param state the found State is returned here if found.
   * @returns true if String is found, false otherwise.
   */
  bool GLGuiStyle::getState(const std::string& stateName, OrxGui::State* state)
  {
    for (unsigned int i = 0; i < GLGUI_STATE_COUNT; ++i)
      if (stateName == OrxGui::StateString[i])
      {
        *state = (OrxGui::State)i;
        return true;
      }
    return false;
  }
}
