/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "md3_model.h"

#include "md3_data.h"
#include "md3_mesh.h"
#include "md3_tag.h"
#include "md3_bone_frame.h"

#include "md3_animation_cfg.h"

#include "material.h"
#include "quaternion.h"

#include "debug.h"

namespace md3
{

  /**
   * md3 model
   */
  MD3Model::MD3Model(std::string filename, float scaling)
  {

    this->autoAssemblePlayerModel(filename, scaling);


    // create the temporary data to work with (interpolation data)
    this->tmpBoneFrame = new MD3BoneFrame();

    this->tmpMesh = new sVec3D*[this->md3Data->header->meshNum];
    for( int i = 0; i < this->md3Data->header->meshNum; i++)
      this->tmpMesh[i] = new sVec3D[this->md3Data->meshes[i]->header->vertexNum];

    this->tmpNormal = new MD3Normal*[this->md3Data->header->meshNum];
    for( int i = 0; i < this->md3Data->header->meshNum; i++)
      this->tmpNormal[i] = new MD3Normal[this->md3Data->meshes[i]->header->vertexNum];

    this->tmpMatrix = new float*[this->md3Data->header->tagNum];
    for( int i = 0; i < this->md3Data->header->tagNum; i++)
      this->tmpMatrix[i] = new float[16];

    this->bDrawBones = false;
    this->bDrawNormals = false;
  }



  MD3Model::~MD3Model()
  {
    delete this->tmpBoneFrame;
//     delete [] this->tmpMesh;

    ///TODO deleting mesh
    ///TODO deleting matrices
  }


  /**
   * auto assemples a player model
   */
  void MD3Model::autoAssemblePlayerModel(std::string filename, float scaling)
  {
    this->md3Data = new MD3Data(filename, filename, scaling);
//     MD3AnimationCfg cfg("/home/boenzlip/tmp/q3/Downloads/MOH/q3mdl-alien3/models/players/alien3/animation.cfg");


    // load torso
//     name = FilenameUtils.getShortFilename(root.loadFilename);
//     if( name.find("lower") == 0 && (tagIndex=root.getTagIndexByName("tag_torso")) != -1) {
//       filename=searchForPath("upper" + FilenameUtils.getLODPostfix(name) + ".md3", false);
//       root=attachModel(root, tagIndex, filename, getInputStreamForPath(filename));
//     }

    // load head
//     name=FilenameUtils.getShortFilename(root.loadFilename);
//     if (name.toLowerCase().startsWith("upper") && (tagIndex=root.getTagIndexByName("tag_head")) != -1) {
//       filename=searchForPath("head" + FilenameUtils.getLODPostfix(name) + ".md3", false);
//       attachModel(root, tagIndex, filename, getInputStreamForPath(filename));
//     }

  }



  void MD3Model::tick(float time)
  {

    this->tick(time, this->md3Data);
  }


  void MD3Model::tick(float time, MD3Data* data)
  {
    // draw the bones if needed
    if( this->bDrawBones)
    {
      // get bone frame, interpolate if necessary
      if( data->animationState.interpolationFraction != 0.0 &&
          data->animationState.currentFrame != data->animationState.nextFrame)
      {
        //interpolate bone frame
        this->tmpBoneFrame = this->interpolateBoneFrame(data->boneFrames[data->animationState.currentFrame],
                                                        data->boneFrames[data->animationState.nextFrame],
                                                        data->animationState.interpolationFraction);
      }
      else
      {
        this->tmpBoneFrame = data->boneFrames[data->animationState.currentFrame];
      }
    }

    //draw all meshes of current frame of this model
    for( int i = 0;  i < data->header->meshNum; i++)
    {
      MD3Mesh* mesh = data->meshes[i];

      // get mesh frame, do interpolation if necessary
      sVec3D* frame;
      if( data->animationState.interpolationFraction != 0.0 &&
          data->animationState.currentFrame != data->animationState.nextFrame)
      {
        //interpolate mesh frame between the 2 current mesh frames
        frame = this->interpolateMeshFrame( data->meshes[data->animationState.currentFrame]->meshFrames,
                                            data->meshes[data->animationState.nextFrame]->meshFrames,
                                            data->animationState.interpolationFraction, mesh, i);
      }
      else
      {
        //no interpolation needed, just draw current frame
        frame = &mesh->meshFrames[data->animationState.currentFrame];
      }
      this->tmpMesh[i] = frame;

      // draw vertex normals if needed
      if( this->bDrawNormals)
      {
        // get vertex normals, interpolate if necessary
        if( data->animationState.interpolationFraction != 0.0 &&
            data->animationState.currentFrame != data->animationState.nextFrame)
        {
          //interpolate vertex normals
          this->interpolateVertexNormals(&mesh->normals[data->animationState.currentFrame],
                                         &mesh->normals[data->animationState.nextFrame],
                                         data->animationState.interpolationFraction, mesh, i);
        }
      }
    }


    // draw all models linked to this model
    std::map<int, MD3Data*>::iterator it = data->sortedMap.begin();
    int i = 0;
    while( it != data->sortedMap.end())
    {
      MD3Data* child = it->second;

      //build transformation array m from matrix, interpolate if necessary
      float* m = new float[16];

      MD3Tag* currFrameTag = data->boneFrames[data->animationState.currentFrame]->tags[child->parentTagIndex];

      if( data->animationState.interpolationFraction != 0.0 &&
          data->animationState.currentFrame != data->animationState.nextFrame) {
        //we need to interpolate
        MD3Tag* nextFrameTag = data->boneFrames[data->animationState.nextFrame]->tags[child->parentTagIndex];
        m = this->interpolateTransformation(currFrameTag, nextFrameTag, data->animationState.interpolationFraction, i);
          }
          else {
            //no interpolation needed, stay with last transformation
            //OpenGL matrix is in column-major order
            m[0] = currFrameTag->matrix[0][0];
            m[1] = currFrameTag->matrix[1][0];
            m[2] = currFrameTag->matrix[2][0];
            m[3] = 0.0f;
            m[4] = currFrameTag->matrix[0][1];
            m[5] = currFrameTag->matrix[1][1];
            m[6] = currFrameTag->matrix[2][1];
            m[7] = 0.0f;
            m[8] = currFrameTag->matrix[0][2];
            m[9] = currFrameTag->matrix[1][2];
            m[10]= currFrameTag->matrix[2][2];
            m[11]= 0.0f;
            m[12] = currFrameTag->position.x;
            m[13] = currFrameTag->position.y;
            m[14] = currFrameTag->position.z;
            m[15] = 1.0f;
          }

          //switch to child coord system
          glPushMatrix();
          glMultMatrixf(m);

          // and draw child
          this->tick(time, child);

          glPopMatrix();
          i++;
    }
  }


  /**
   * this draws the md3 model
   */
  void MD3Model::draw() const
  {
    //draw current bone frame
    this->draw(this->md3Data);
  }


  /**
   * draw the md3model
   * @param data: the data to be drawn
   */
  void MD3Model::draw(MD3Data* data) const
  {

    // draw the bones if needed
    if( this->bDrawBones)
    {
      // get bone frame, interpolate if necessary
      if( data->animationState.interpolationFraction != 0.0 &&
          data->animationState.currentFrame != data->animationState.nextFrame) {
        //interpolate bone frame
        this->drawBoneFrame(this->tmpBoneFrame);
      }
      else {
        //stick with current bone frame
        this->drawBoneFrame(data->boneFrames[data->animationState.currentFrame]);
      }
    }


    //draw all meshes of current frame of this model
    for( int i = 0;  i < data->header->meshNum; i++)
    {
      MD3Mesh* mesh = data->meshes[i];

//       gl.glBlendFunc(mesh.GLSrcBlendFunc, mesh.GLDstBlendFunc);
//       gl.glDepthMask(mesh.GLDepthMask);

      if( mesh->header->textureNum > 0 && &mesh->material[0] != NULL)
        mesh->material[0].select();

      // get mesh frame, do interpolation if necessary
      sVec3D* frame = this->tmpMesh[i];

      this->drawMesh(mesh, frame);


      // draw vertex normals if needed
      if( this->bDrawNormals) {
        // get vertex normals, interpolate if necessary
        if( data->animationState.interpolationFraction != 0.0 &&
            data->animationState.currentFrame != data->animationState.nextFrame) {
          //interpolate vertex normals
          this->drawVertexNormals(frame, this->tmpNormal[i]);
            }
            else {
          //stick with current vertex normals
              this->drawVertexNormals(frame, &mesh->normals[data->animationState.currentFrame]);
            }
      }
    }


    // draw all models linked to this model
    int i = 0;
    std::map<int, MD3Data*>::iterator it = data->sortedMap.begin();
    while( it != data->sortedMap.end())
    {
      MD3Data* child = it->second;

      //build transformation array m from matrix, interpolate if necessary
      float* m = new float[16];

      MD3Tag* currFrameTag = data->boneFrames[data->animationState.currentFrame]->tags[child->parentTagIndex];

      if( data->animationState.interpolationFraction != 0.0 &&
          data->animationState.currentFrame != data->animationState.nextFrame) {
        //we need to interpolate
        m = this->tmpMatrix[i];
      }
      else {
        //no interpolation needed, stay with last transformation
        //OpenGL matrix is in column-major order
        m[0] = currFrameTag->matrix[0][0];
        m[1] = currFrameTag->matrix[1][0];
        m[2] = currFrameTag->matrix[2][0];
        m[3] = 0.0f;
        m[4] = currFrameTag->matrix[0][1];
        m[5] = currFrameTag->matrix[1][1];
        m[6] = currFrameTag->matrix[2][1];
        m[7] = 0.0f;
        m[8] = currFrameTag->matrix[0][2];
        m[9] = currFrameTag->matrix[1][2];
        m[10]= currFrameTag->matrix[2][2];
        m[11]= 0.0f;
        m[12] = currFrameTag->position.x;
        m[13] = currFrameTag->position.y;
        m[14] = currFrameTag->position.z;
        m[15] = 1.0f;
      }

      //switch to child coord system
      glPushMatrix();
      glMultMatrixf(m);

      // and draw child
      this->draw(child);

      glPopMatrix();
      i++;
    }
  }


  /**
   * draws the mesh
   */
  void MD3Model::drawMesh(MD3Mesh* mesh, sVec3D* frame) const
  {
    glColor3f(1.0f, 1.0f, 1.0f);

    glBegin( GL_TRIANGLES);
    glScalef( 0.7f , 0.7f, 0.7f );

    Vector tmpVec1, tmpVec2;

    // upload all triangles in the frame to OpenGL
    for( int t = 0; t < mesh->header->triangleNum; t++)
    {
      // calc normal vector
      tmpVec1.x = frame[mesh->triangles[t].vertexOffset[1]][0] - frame[mesh->triangles[t].vertexOffset[0]][0];
      tmpVec1.y = frame[mesh->triangles[t].vertexOffset[1]][1] - frame[mesh->triangles[t].vertexOffset[0]][1];
      tmpVec1.z = frame[mesh->triangles[t].vertexOffset[1]][2] - frame[mesh->triangles[t].vertexOffset[0]][2];

      tmpVec2.x = frame[mesh->triangles[t].vertexOffset[2]][0] - frame[mesh->triangles[t].vertexOffset[0]][0];
      tmpVec2.y = frame[mesh->triangles[t].vertexOffset[2]][1] - frame[mesh->triangles[t].vertexOffset[0]][1];
      tmpVec2.z = frame[mesh->triangles[t].vertexOffset[2]][2] - frame[mesh->triangles[t].vertexOffset[0]][2];

      Vector normal = tmpVec1.cross(tmpVec2);
      normal.normalize();

//       PRINTF(0)("normal: %f, %f, %f\n", normal.x, normal.y, normal.z);

      glNormal3f(normal.x, normal.y, normal.z);
      glTexCoord2fv( mesh->texVecs[mesh->triangles[t].vertexOffset[0]].textureCoord);
      glVertex3f( frame[mesh->triangles[t].vertexOffset[0]][0],
                  frame[mesh->triangles[t].vertexOffset[0]][2],
                  frame[mesh->triangles[t].vertexOffset[0]][1]);

      glNormal3f(normal.x, normal.y, normal.z);
      glTexCoord2fv( mesh->texVecs[mesh->triangles[t].vertexOffset[1]].textureCoord);
      glVertex3f( frame[mesh->triangles[t].vertexOffset[1]][0],
                  frame[mesh->triangles[t].vertexOffset[1]][2],
                  frame[mesh->triangles[t].vertexOffset[1]][1]);

      glNormal3f(normal.x, normal.y, normal.z);
      glTexCoord2fv( mesh->texVecs[mesh->triangles[t].vertexOffset[2]].textureCoord);
      glVertex3f( frame[mesh->triangles[t].vertexOffset[2]][0],
                  frame[mesh->triangles[t].vertexOffset[2]][2],
                  frame[mesh->triangles[t].vertexOffset[2]][1]);
    }
    glEnd();

  }


  /**
   *  drawo vertex normals
   */
  void MD3Model::drawVertexNormals(sVec3D* frame, MD3Normal* normals) const
  {}


  /**
   * draw bone frame
   */
  void MD3Model::drawBoneFrame(MD3BoneFrame* frame) const
  {
    float x1 = frame->mins.x;
    float y1 = frame->mins.y;
    float z1 = frame->mins.z;
    float x2 = frame->maxs.x;
    float y2 = frame->maxs.y;
    float z2 = frame->maxs.z;

    glPushAttrib(GL_TEXTURE_2D);
    glPushAttrib(GL_LIGHTING);

    glColor3f(1.0f,0.0f,0.0f);
    glPointSize(6.0f);

    glBegin(GL_POINTS);
    glVertex3f(frame->position.x, frame->position.y, frame->position.z);
    glEnd();
    glPointSize(1.0f);

    glColor3f(0.0f,1.0f,0.0f);
    glBegin(GL_LINE_LOOP);
    glVertex3f(x1,y1,z1);
    glVertex3f(x1,y1,z2);
    glVertex3f(x1,y2,z2);
    glVertex3f(x1,y2,z1);
    glEnd();

    glBegin(GL_LINE_LOOP);
    glVertex3f(x2,y2,z2);
    glVertex3f(x2,y1,z2);
    glVertex3f(x2,y1,z1);
    glVertex3f(x2,y2,z1);
    glEnd();

    glBegin(GL_LINES);
    glVertex3f(x1,y1,z1);
    glVertex3f(x2,y1,z1);

    glVertex3f(x1,y1,z2);
    glVertex3f(x2,y1,z2);

    glVertex3f(x1,y2,z2);
    glVertex3f(x2,y2,z2);

    glVertex3f(x1,y2,z1);
    glVertex3f(x2,y2,z1);
    glEnd();

     glPopAttrib();
     glPopAttrib();
  }


  /**
   *  interpolate bone frame
   * @param currBoneFrame Start bone frame.
   * @param nextBoneFrame End bone frame.
   * @param frac Interpolation fraction, in [0,1].
   */
  MD3BoneFrame* MD3Model::interpolateBoneFrame(MD3BoneFrame* currBoneFrame, MD3BoneFrame* nextBoneFrame, float frac)
  {
    this->tmpBoneFrame->mins.x      = (1.0f - frac) * currBoneFrame->mins.x       + frac * nextBoneFrame->mins.x;
    this->tmpBoneFrame->maxs.x      = (1.0f - frac) * currBoneFrame->maxs.x       + frac * nextBoneFrame->maxs.x;
    this->tmpBoneFrame->position.x  = (1.0f - frac) * currBoneFrame->position.x   + frac * nextBoneFrame->position.x;
    this->tmpBoneFrame->mins.y      = (1.0f - frac) * currBoneFrame->mins.y       + frac * nextBoneFrame->mins.y;
    this->tmpBoneFrame->maxs.y      = (1.0f - frac) * currBoneFrame->maxs.y       + frac * nextBoneFrame->maxs.y;
    this->tmpBoneFrame->position.y  = (1.0f - frac) * currBoneFrame->position.y   + frac * nextBoneFrame->position.y;
    this->tmpBoneFrame->mins.z      = (1.0f - frac) * currBoneFrame->mins.z       + frac * nextBoneFrame->mins.z;
    this->tmpBoneFrame->maxs.z      = (1.0f - frac) * currBoneFrame->maxs.z       + frac * nextBoneFrame->maxs.z;
    this->tmpBoneFrame->position.z  = (1.0f - frac) * currBoneFrame->position.z   + frac * nextBoneFrame->position.z;

    return this->tmpBoneFrame;
  }



  /**
   * interpolate mesh frame
   */
  sVec3D* MD3Model::interpolateMeshFrame(sVec3D* currMeshFrame, sVec3D* nextMeshFrame, float frac, MD3Mesh* mesh, int i)
  {
    int vertexNum = mesh->header->vertexNum;

    // calc interpolated vertices
    for( int t = 0; t < vertexNum * 3.0f; t++)
    {

//       PRINTF(0)("tmp mesh: %f, %f, %f\n", this->tmpMesh[i][t][0], this->tmpMesh[i][t][1], this->tmpMesh[i][t][2]);

      this->tmpMesh[i][t][0]  = (1.0f - frac)   * currMeshFrame[t][0]  + frac * nextMeshFrame[t][0];
      this->tmpMesh[i][t][1]  = (1.0f - frac)   * currMeshFrame[t][1]  + frac * nextMeshFrame[t][1];
      this->tmpMesh[i][t][2]  = (1.0f - frac)   * currMeshFrame[t][2]  + frac * nextMeshFrame[t][2];
    }

    return this->tmpMesh[i];
  }


  /**
   * interpolate vertex normal
   */
  MD3Normal* MD3Model::interpolateVertexNormals(MD3Normal* currNormals, MD3Normal* nextNormals, float frac, MD3Mesh* mesh, int i)
  {
    for( int j = 0; j < mesh->header->vertexNum; j++)
    {
      this->tmpNormal[i][j].vertexNormal[0] = (int)((1.0f - frac) * currNormals[j].vertexNormal[0] + frac * nextNormals[j].vertexNormal[0]);
      this->tmpNormal[i][j].vertexNormal[1] = (int)((1.0f - frac) * currNormals[j].vertexNormal[1] + frac * nextNormals[j].vertexNormal[1]);
    }

    return this->tmpNormal[i];
  }


  /**
   * interpolate transformation
   */
  float* MD3Model::interpolateTransformation(MD3Tag* currFrameTag, MD3Tag* nextFrameTag, float frac, int i)
  {
    // interpolate position
    Vector interpolatedPosition = currFrameTag->position * (1.0f - frac) + nextFrameTag->position * frac;


    // interpolate rotation matrix
    float  currRot[4][4];
    float  nextRot[4][4];
    float  interpolatedMatrix[4][4];

    Quaternion currQuat(currFrameTag->matrix); currQuat.matrix(currRot);
    Quaternion nextQuat(nextFrameTag->matrix); nextQuat.matrix(nextRot);

    Quaternion interpolatedQuat = Quaternion::quatSlerp(currQuat, nextQuat, frac); interpolatedQuat.matrix(interpolatedMatrix);

    // quaternion code is column based, so use transposed matrix when spitting out to gl
    this->tmpMatrix[i][0] = interpolatedMatrix[0][0];
    this->tmpMatrix[i][4] = interpolatedMatrix[1][0];
    this->tmpMatrix[i][8] = interpolatedMatrix[2][0];
    this->tmpMatrix[i][12] = interpolatedPosition.x;
    this->tmpMatrix[i][1] = interpolatedMatrix[0][1];
    this->tmpMatrix[i][5] = interpolatedMatrix[1][1];
    this->tmpMatrix[i][9] = interpolatedMatrix[2][1];
    this->tmpMatrix[i][13] = interpolatedPosition.y;
    this->tmpMatrix[i][2] = interpolatedMatrix[0][2];
    this->tmpMatrix[i][6] = interpolatedMatrix[1][2];
    this->tmpMatrix[i][10]= interpolatedMatrix[2][2];
    this->tmpMatrix[i][14] = interpolatedPosition.z;
    this->tmpMatrix[i][3] = 0.0f;
    this->tmpMatrix[i][7] = 0.0f;
    this->tmpMatrix[i][11]= 0.0f;
    this->tmpMatrix[i][15] = 1.0f;

    return this->tmpMatrix[i];

  }


}
