/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

#include "SpaceShip.h"

#include <BulletDynamics/Dynamics/btRigidBody.h>

#include "core/CoreIncludes.h"
#include "core/ConfigValueIncludes.h"
#include "core/Template.h"
#include "core/XMLPort.h"
#include "items/Engine.h"
#include "graphics/Camera.h"
#include "CameraManager.h"
#include "util/Math.h"

namespace orxonox
{
    const float orientationGain = 100;
    CreateFactory(SpaceShip);

    SpaceShip::SpaceShip(BaseObject* creator) : Pawn(creator)
    {
        RegisterObject(SpaceShip);

        this->primaryThrust_  = 100;
        this->auxilaryThrust_ =  30;
        this->rotationThrust_ =  10;

        this->localLinearAcceleration_.setValue(0, 0, 0);
        this->localAngularAcceleration_.setValue(0, 0, 0);
        this->bBoost_ = false;
        this->steering_ = Vector3::ZERO;
        this->engine_ = 0;

        this->boostPower_ = 10.0f;
        this->initialBoostPower_ = 10.0f;
        this->boostRate_ = 5.0;
        this->boostPowerRate_ = 1.0;
        this->boostCooldownDuration_ = 5.0;
        this->bBoostCooldown_ = false;

        this->bInvertYAxis_ = false;

        this->setDestroyWhenPlayerLeft(true);

        // SpaceShip is always a physical object per default
        // Be aware of this call: The collision type legality check will not reach derived classes!
        this->setCollisionType(WorldEntity::Dynamic);
        // Get notification about collisions
        this->enableCollisionCallback();

        this->setConfigValues();
        this->registerVariables();
        
        Camera* c = CameraManager::getInstance().getActiveCamera();
        this->cameraOriginalPosition = c->getPosition();
        this->cameraOriginalOrientation = c->getOrientation();

        this->shakeFrequency_ = 15;
        this->shakeAmplitude_ = 5;
        this->shakeDt_ = 0;
    }

    SpaceShip::~SpaceShip()
    {
        if (this->isInitialized() && this->engine_)
            this->engine_->destroy();
    }

    void SpaceShip::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(SpaceShip, XMLPort, xmlelement, mode);

        XMLPortParam(SpaceShip, "engine",            setEngineTemplate,    getEngineTemplate,    xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "primaryThrust",  primaryThrust_,  xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "auxilaryThrust", auxilaryThrust_, xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "rotationThrust", rotationThrust_, xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "boostPower", initialBoostPower_, xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "boostPowerRate", boostPowerRate_, xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "boostRate", boostRate_, xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "boostCooldownDuration", boostCooldownDuration_, xmlelement, mode);
        XMLPortParamVariable(SpaceShip, "shakeFrequency", shakeFrequency_, xmlelement, mode);
    }

    void SpaceShip::registerVariables()
    {
        registerVariable(this->primaryThrust_,  VariableDirection::ToClient);
        registerVariable(this->auxilaryThrust_, VariableDirection::ToClient);
        registerVariable(this->rotationThrust_, VariableDirection::ToClient);
        registerVariable(this->boostPower_, VariableDirection::ToClient);
        registerVariable(this->boostPowerRate_, VariableDirection::ToClient);
        registerVariable(this->boostRate_, VariableDirection::ToClient);
        registerVariable(this->boostCooldownDuration_, VariableDirection::ToClient);
        registerVariable(this->shakeFrequency_, VariableDirection::ToClient);
    }

    void SpaceShip::setConfigValues()
    {
        SetConfigValue(bInvertYAxis_, false).description("Set this to true for joystick-like mouse behaviour (mouse up = ship down).");
    }

    bool SpaceShip::isCollisionTypeLegal(WorldEntity::CollisionType type) const
    {
        if (type != WorldEntity::Dynamic)
        {
            CCOUT(1) << "Error: Cannot tell a SpaceShip not to be dynamic! Ignoring." << std::endl;
            assert(false); // Only in debug mode
            return false;
        }
        else
            return true;
    }

    void SpaceShip::tick(float dt)
    {
        SUPER(SpaceShip, tick, dt);

        if (this->hasLocalController())
        {

/*
            this->localLinearAcceleration_.setX(this->localLinearAcceleration_.x() * getMass() * this->auxilaryThrust_);
            this->localLinearAcceleration_.setY(this->localLinearAcceleration_.y() * getMass() * this->auxilaryThrust_);
            if (this->localLinearAcceleration_.z() > 0)
                this->localLinearAcceleration_.setZ(this->localLinearAcceleration_.z() * getMass() * this->auxilaryThrust_);
            else
                this->localLinearAcceleration_.setZ(this->localLinearAcceleration_.z() * getMass() * this->primaryThrust_);
            this->physicalBody_->applyCentralForce(physicalBody_->getWorldTransform().getBasis() * this->localLinearAcceleration_);
            this->localLinearAcceleration_.setValue(0, 0, 0);
*/
            if (!this->isInMouseLook())
            {
                this->localAngularAcceleration_ *= this->getLocalInertia() * this->rotationThrust_;
                this->physicalBody_->applyTorque(physicalBody_->getWorldTransform().getBasis() * this->localAngularAcceleration_);
            }

            this->localAngularAcceleration_.setValue(0, 0, 0);

            if(!this->bBoostCooldown_ && this->boostPower_ < this->initialBoostPower_)
            {
                this->boostPower_ += this->boostPowerRate_*dt;
            }

            if(this->bBoost_)
            {
                this->boostPower_ -=this->boostRate_*dt;
                if(this->boostPower_ <= 0.0f)
                {
                    this->boost(false);
                    this->bBoostCooldown_ = true;
                    this->timer_.setTimer(this->boostCooldownDuration_, false, createExecutor(createFunctor(&SpaceShip::boostCooledDown, this)));

                }

                shakeCamera(dt);    
            }
        }
    }

    void SpaceShip::moveFrontBack(const Vector2& value)
    {
        this->localLinearAcceleration_.setZ(this->localLinearAcceleration_.z() - value.x);
        this->steering_.z = -value.x;
    }

    void SpaceShip::moveRightLeft(const Vector2& value)
    {
        this->localLinearAcceleration_.setX(this->localLinearAcceleration_.x() + value.x);
        this->steering_.x = value.x;
    }

    void SpaceShip::moveUpDown(const Vector2& value)
    {
        this->localLinearAcceleration_.setY(this->localLinearAcceleration_.y() + value.x);
        this->steering_.y = value.x;
    }

    void SpaceShip::rotateYaw(const Vector2& value)
    {
        this->localAngularAcceleration_.setY(this->localAngularAcceleration_.y() + value.x);

        Pawn::rotateYaw(value);
    }

    void SpaceShip::rotatePitch(const Vector2& value)
    {
        this->localAngularAcceleration_.setX(this->localAngularAcceleration_.x() + value.x);

        Pawn::rotatePitch(value);
    }

    void SpaceShip::rotateRoll(const Vector2& value)
    {
        this->localAngularAcceleration_.setZ(this->localAngularAcceleration_.z() + value.x);

        Pawn::rotateRoll(value);
    }

    void SpaceShip::fire()
    {
    }

    /**
    @brief
        Starts or stops boosting.
    @param bBoost
        Whether to start or stop boosting.
    */
    void SpaceShip::boost(bool bBoost)
    {
        if(bBoost && !this->bBoostCooldown_)
        {
            //COUT(0) << "Boost startet!\n";
            this->bBoost_ = true;
        }
        if(!bBoost)
        {
            //COUT(0) << "Boost stoppt\n";
            this->resetCamera();
            this->bBoost_ = false;
        }
    }
    
    void SpaceShip::boostCooledDown(void)
    {
        this->bBoostCooldown_ = false;
    }
    
    void SpaceShip::shakeCamera(float dt)
    {
        //make sure the ship is only shaking if it's moving
        if (this->getVelocity().squaredLength() > 80)
        {
            this->shakeDt_ += dt;
    
            int frequency = this->shakeFrequency_ * (this->getVelocity().squaredLength());
    
            if (this->shakeDt_ >= 1 /(frequency))
            {
                this->shakeDt_ -= 1/(frequency);
            }
    
            Degree angle = Degree(sin(this->shakeDt_ * 2* math::pi * frequency) * this->shakeAmplitude_);
    
            //COUT(0) << "Angle: " << angle << std::endl;
            Camera* c = this->getCamera();

            //Shaking Camera effect
            if (c != 0)
            {
                c->setOrientation(Vector3::UNIT_X, angle);
            }
        }
    }
    
    void SpaceShip::resetCamera()
    {
    
        //COUT(0) << "Resetting camera\n";
        Camera *c = this->getCamera();
    
        if (c == 0)
        {
            COUT(2) << "Failed to reset camera!";
            return;
        }
    
        shakeDt_ = 0;
        //
        c->setPosition(this->cameraOriginalPosition);
        c->setOrientation(this->cameraOriginalOrientation);
    }

    void SpaceShip::loadEngineTemplate()
    {
        if (!this->enginetemplate_.empty())
        {
            Template* temp = Template::getTemplate(this->enginetemplate_);

            if (temp)
            {
                Identifier* identifier = temp->getBaseclassIdentifier();

                if (identifier)
                {
                    BaseObject* object = identifier->fabricate(this);
                    this->engine_ = orxonox_cast<Engine*>(object);

                    if (this->engine_)
                    {
                        this->engine_->addTemplate(temp);
                        this->engine_->addToSpaceShip(this);
                    }
                    else
                    {
                        object->destroy();
                    }
                }
            }
        }
    }

    void SpaceShip::setEngine(Engine* engine)
    {
        this->engine_ = engine;
        if (engine && engine->getShip() != this)
            engine->addToSpaceShip(this);
    }

    std::vector<PickupCarrier*>* SpaceShip::getCarrierChildren(void) const
    {
        std::vector<PickupCarrier*>* list = new std::vector<PickupCarrier*>();
        list->push_back(this->engine_);
        return list;
    }
    

}
